/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.mdconfig;

import java.io.File;
import java.io.Serializable;
import java.util.*;

import javax.swing.ImageIcon;

import org.dom4j.*;
import org.dom4j.io.SAXReader;

import ch.docuteam.docudarc.ead.C;
import ch.docuteam.docutools.out.Logger;


/**
 * This class represents a level of description in a METS file.
 * The levels are initialized through the file "./config/levels.xml".
 * <p>
 * A level has a name, an order number, and optionally an icon.
 * <p>
 * A level has a list of levels that are allowed as its sub-levels, and one level that is the default sub-level.
 * <p>
 * A level has a list of <a href="./LevelMetadataElement.html">LevelMetadataElements</a>
 * (actually two lists: a Map for quick access and a List&lt;LevelOfDescription&gt; for displaying them in a GUI in their correct order).
 *
 * @author denis
 *
 */
public class LevelOfDescription implements Serializable
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	The default file for initializing levels.
	static private final String							DefaultXMLFileName = "./config/levels.xml";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	The currently used file for initializing levels.
	static private String								InitializationFilePath = DefaultXMLFileName;

	//	A Map containing all LevelOfDescriptions. Access key is the level name.
	static private Map<String, LevelOfDescription>		All;

	//	A List containing all LevelOfDescriptions in their correct order.
	static private List<LevelOfDescription>				AllOrdered;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String										name;
	private Integer										order;

	/**
	 * The icon used for this Level in a GUI (might be null).
	 */
	private ImageIcon									icon;

	/**
	 * Set containing this level's allowed sublevel names (might be empty).
	 */
	private Set<String>									allowedSubLevelNames = new HashSet<String>();

	/**
	 * The default level name when inserting a new subelement (might be null):
	 */
	private String										defaultSubLevelName;

	/**
	 * Map containing this level's LevelMetadataElements.
	 * Key is the LevelMetadataElement's accessorName, value is the LevelMetadataElement itself.
	 */
	private Map<String, LevelMetadataElement>			levelMetadataElements = new HashMap<String, LevelMetadataElement>();

	/**
	 * Vector containing this level's LevelMetadataElements in their correct order.
	 */
	private List<LevelMetadataElement>					levelMetadataElementsOrdered = new Vector<LevelMetadataElement>();

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	/**
	 * Constructor used by LevelOfDescriptionSet, to generate Levels that don't belong to the standard set of Levels specified in levels.xml.
	 * @param name
	 */
	LevelOfDescription(String name, int order)
	{
		this.name = name;
		this.order = order;

		Logger.debug("Created: " + this);
	}

	//	========	Constructors Private	=======================================================

	/**
	 * Private Constructor to generate Levels that belong to the standard set of Levels specified in levels.xml.
	 * @param name
	 */
	private LevelOfDescription(String name, String iconFileName, String allowedSubLevelNames)
	{
		this.name = name;
		this.order = All.size();

		All.put(name, this);
		AllOrdered.add(this);

		if (iconFileName != null && new File(iconFileName).exists())
			this.icon = new ImageIcon(iconFileName);

		if (allowedSubLevelNames != null && !allowedSubLevelNames.trim().isEmpty())
		{
			String[] allAllowedSubLevelNames = allowedSubLevelNames.split("\\s+");
			this.defaultSubLevelName = allAllowedSubLevelNames[0];
			for (String subLevelName: allAllowedSubLevelNames)		this.allowedSubLevelNames.add(subLevelName);
		}

		Logger.debug("Created standard Level: " + this);
	}

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static public void setInitializationFilePath(String initializationFilePath)
	{
		InitializationFilePath = initializationFilePath;
		All = null;
		AllOrdered = null;
		MetadataElement.clear();
		//	LevelsOfDescription and MetadataElements will be (re-)initialized the next time they are accessed.
	}

	//	--------		Accessing			-------------------------------------------------------

	static public LevelOfDescription getUndefined()
	{
		initializeIfNecessary();
		return AllOrdered.get(0);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static private void initialize()
	{
		Logger.debug("Initializing LevelOfDescriptions from file: " + InitializationFilePath);

		MetadataElement.clear();

		All = new HashMap<String, LevelOfDescription>();
		AllOrdered = new Vector<LevelOfDescription>();

		org.dom4j.Document levelsConfigFile;
		try
		{
			levelsConfigFile = new SAXReader().read(InitializationFilePath);

			Element rootElement = levelsConfigFile.getRootElement();

			Logger.debug("Reading optional AllowedValuesSeparator:");
			Node n = rootElement.selectSingleNode("./LEVELS:MetadataElements/LEVELS:AllowedValuesSeparator");
			if (n != null)		MetadataElement.setAllowedValuesSeparator(n.getText());

//			ExceptionCollector.clear();		//	ToDo: Nested Exception Collectors?

			Logger.debug("Reading MetadataElements:");
			for (Object e: rootElement.selectNodes("./LEVELS:MetadataElements/LEVELS:MetadataElement"))
			{
				Element mde = (Element)e;
				try
				{
					new MetadataElement(C.class, mde.attributeValue("accessorNameID"), mde.attributeValue("validatorClassName"), mde.attributeValue("defaultExpression"), mde.attributeValue("allowedValues"));
				}
				catch (Exception ex)
				{
					ch.docuteam.docutools.exception.Exception.remember(ex);
				}
			}
			Logger.debug("Reading MetadataElements finished.");

//			if (!ExceptionCollector.isEmpty())		throw new ExceptionCollectorException();		//	ToDo: Nested Exception Collectors?

			Logger.debug("Reading Levels:");
			for (Object l: rootElement.selectNodes("./LEVELS:Levels/LEVELS:Level"))
			{
				Element le = (Element)l;
				LevelOfDescription level = new LevelOfDescription(le.attributeValue("nameID"), le.attributeValue("iconFileName"), le.attributeValue("allowedSublevelNameRefs"));

				for (Object o: le.selectNodes("./LEVELS:LevelMetadataElement"))
				{
					Element lme = (Element)o;
					LevelMetadataElement levelMetadataElement = new LevelMetadataElement(lme.attributeValue("accessorNameRef"), lme.attributeValue("isMandatory"), lme.attributeValue("isAlwaysDisplayed"), lme.attributeValue("isRepeatable"), lme.attributeValue("isReadOnly"), lme.attributeValue("keepInTemplate"), lme.attributeValue("displayRows"));
					level.levelMetadataElements.put(levelMetadataElement.getId(), levelMetadataElement);
					level.levelMetadataElementsOrdered.add(levelMetadataElement);
				}
			}
			Logger.debug("Reading Levels finished.");
		}
		catch (DocumentException ex)
		{
			ex.printStackTrace();
		}
//		catch (ExceptionCollectorException ex)
//		{
//			ex.printStackTrace();
//		}
	}

	//	--------		Accessing			-------------------------------------------------------

	/**
	 * Return the Levels list in the order the levels were inserted.
	 * Package visibility because this method should be used only by LevelOfDescriptionSet.
	 */
	static List<LevelOfDescription> getAll()
	{
		initializeIfNecessary();
		return AllOrdered;
	}

	//	--------		Inquiring			-------------------------------------------------------

	/**
	 * This method has package visibility because it can be called by MetadataElement.
	 */
	static void initializeIfNecessary()
	{
		if (All == null)		initialize();
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public String getName()
	{
		return this.name;
	}

	public Integer getOrder()
	{
		return this.order;
	}

	/**
	 * The ImageIcon returned may be null
	 * @return
	 */
	public ImageIcon getIcon()
	{
		return this.icon;
	}


	public Set<String> getAllowedSubLevelNames()
	{
		return this.allowedSubLevelNames;
	}

	public LevelOfDescription getDefaultSubLevel()
	{
		return (this.defaultSubLevelName != null)? All.get(this.defaultSubLevelName): getUndefined();
	}

	public Map<String, LevelMetadataElement> getDynamicMetadataElements()
	{
		return this.levelMetadataElements;
	}


	public LevelMetadataElement getDynamicMetadataElement(String accessorName)
	{
		if (!this.levelMetadataElements.containsKey(accessorName))		throw new IllegalArgumentException("Bad metadata name: <" + accessorName + "> in Level: " + this);

		return this.levelMetadataElements.get(accessorName);
	}

	public List<LevelMetadataElement> getDynamicMetadataElementsOrdered()
	{
		return this.levelMetadataElementsOrdered;
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean allowsSubLevel(String name)
	{
		return this.allowedSubLevelNames.contains(name);
	}

	public boolean allowsSubLevel(LevelOfDescription level)
	{
		return this.allowedSubLevelNames.contains(level.name);
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return this.order + ":" + this.name;
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================
}

