/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.mets;

import java.io.*;
import java.util.*;

import org.dom4j.*;
import org.dom4j.io.*;

import ch.docuteam.docudarc.common.DocumentAbstract;
import ch.docuteam.docudarc.ead.MetadataElementInstance;
import ch.docuteam.docudarc.exceptions.*;
import ch.docuteam.docudarc.mets.amdsec.AMDSection;
import ch.docuteam.docudarc.mets.dmdsec.DMDSectionAbstract;
import ch.docuteam.docudarc.mets.filesec.FileSection;
import ch.docuteam.docudarc.mets.metshdr.Header;
import ch.docuteam.docudarc.mets.structmap.*;
import ch.docuteam.docudarc.mets.structmap.NodeAbstract.SubmitStatus;
import ch.docuteam.docudarc.sa.SubmissionAgreement;
import ch.docuteam.docudarc.util.RosettaConverter;
import ch.docuteam.docutools.exception.*;
import ch.docuteam.docutools.exception.Exception;
import ch.docuteam.docutools.file.FileUtil;
import ch.docuteam.docutools.file.Zipper;
import ch.docuteam.docutools.file.exception.FileUtilException;
import ch.docuteam.docutools.file.exception.FileUtilExceptionListException;
import ch.docuteam.docutools.os.OperatingSystem;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.out.Tracer;
import ch.docuteam.docutools.string.DateFormatter;


/**
 * An instance of this class is the java-equivalent of a METS-file.
 * It contains
 * <ul>
 * <li>a <a href="./metshdr/Header.html">Header</a></li>
 * <li>an <a href="./amdsec/AMDSection.html">AMDSection</a></li>
 * <li>a <a href="./filesec/FileSection.html">FileSection</a></li>
 * <li>and a <a href="./structmap/StructureMap.html">StructureMap</a></li>
 * </ul>
 * <p>
 * This class inserts Exceptions into the <a href="../../docutools/exception/ExceptionCollector.html">ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>When some files don't exist in the file system but should exist according to the METS file</li>
 * </ul>
 * @author denis
 */
@SuppressWarnings("serial")
public class Document extends DocumentAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Final Static Public		=======================================================

	static final public String			DefaultMetsFileName = "mets.xml";
	static final public String			LockFileSuffix = ".lock";

	//	========	Final Static Private	=======================================================

	static final protected String		WorkingFolderPrefix = "METSDocument_read_";
	static final protected String		WorkingFolderPrefixReadZip = "METSDocument_readZIP_";
	static final protected String		WorkingFolderPrefixCreateZip = "METSDocument_createZIP_";

	static final protected String		DefaultMETS_SIPTemplateFile = "resources/templates/mets_sip.xml";

	static final protected Integer		DefaultKeepBackupsCount = 10;

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static protected String				METS_SIPTemplateFile = DefaultMETS_SIPTemplateFile;

	/**
	 * If the backup folder is null, backups are created in the folder the SIP is in; otherwise in the specified backup folder (which can be either absolute or relative).
	 */
	static protected String				BackupFolder = null;

	/**
	 * How many backups to keep.
	 * 		0:		keep none
	 * 		3:		keep 3
	 * 		< 0:	is illegal and throws an exception
	 */
	static protected Integer			KeepBackupsCount = DefaultKeepBackupsCount;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * The mode tells in how far the SIP can be modified.
	 */
	protected Mode						mode = Mode.Undefined;

	/**
	 * The folder where the original SIP is located. It can be both relative or absolute. It is never null.
	 */
	protected String					originalSipFolder;

	/**
	 * True when the current SIP is a working copy, false when working directly on a SIP.
	 * In the latter case, all file operations are disabled.
	 * NOTE: When the original SIP is a ZIP archive, a working copy always exists, therefore isWorkingCopy is always true.
	 * Nevertheless file operations may be disabled when the SIP was opened in ReadOnly or NoFileOps mode.
	 */
	protected boolean					isWorkingCopy = true;

	/**
	 * The name of the user who locked this SIP. If null, the current user locked this SIP.
	 */
	protected String					lockedBy;


	//	Root element properties:
	protected String					namespaceMETS;
	protected String					namespacePREMIS;
	protected String					profile;
	protected String					label;
	protected String					type;
	protected String					saId;
	protected String					dssId;
	protected String					objId;


	/**
	 * The METS Header of this document.
	 */
	protected Header					header;

	/**
	 * Map containing the optional DMDSections. Key is the DMDSection's id, value is the DMDSection itself.
	 */
	protected Map<String, DMDSectionAbstract>
										dmdSections = new HashMap<String, DMDSectionAbstract>();

	/**
	 * The AMD Section of this document.
	 */
	protected AMDSection				amdSection;

	/**
	 * The File Section of this document.
	 */
	protected FileSection				fileSection;

	/**
	 * The Structure Map of this document.
	 */
	protected StructureMap				structureMap;

	/**
	 * Who is currently using this Document?
	 */
	protected String					currentOperatorName;

	/**
	 * The submission agreement, initialized lazily only when needed.
	 */
	protected SubmissionAgreement		submissionAgreement;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * For testing.
	 *
	 * @param args not needed
	 */
	public static void main(String[] args)
	{
		Document doc = null;

		try
		{
			//	----------	Reading a SIP:
//			doc = Document.openReadWriteFilesLocked("files/ch-001194-4_459 Nested", "Document.main");
//			doc = Document.openReadWrite("files/ch-001194-4_459 Nested", "Document.main");
//			doc = Document.openReadOnly("files/ch-001194-4_459.zip", "Document.main");
//			doc = Document.openReadOnly("files/ch-001194-4_459", "Document.main");
//			doc = Document.openReadOnly(OperatingSystem.userHome() + "/Desktop/SIPs/BARSIP1", "Document.main");
//			doc = Document.openReadOnly(OperatingSystem.userHome() + "/Desktop/MetsWithSingleFile.zip", "Document.main");

			//	----------	Creating a new folder:
//			NodeFolder newNodeFolder = ((NodeFolder)doc.getStructureMap().getRoot()).createNewFolder("NewFolder");
//			Tracer.trace(newNodeFolder);

			//	----------	Adding a file:
//			NodeFile newNodeFile = (NodeFile)newNodeFolder.insertFileOrFolder("files/Test.tif");
//			Tracer.trace(newNodeFile);

			//	----------	Creating new SIPs:
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new", "files/Handbuch_Arena.pdf", "sa_all-formats-01", "dss-01", "Document.main");	//	Create a new Document with a file as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new.zip", "files/Handbuch_Arena.pdf", "sa_all-formats-01", "dss-01", "Document.main");	//	Create a new ZIP Document with a file as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new", "files/Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new Document with a folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/created/new.zip", "files/Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with a folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/SIPs/Empty Folder", OperatingSystem.userHome() + "/Desktop/Empty Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new Document with an empty folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/SIPs/Empty Folder.zip", OperatingSystem.userHome() + "/Desktop/Empty Folder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with an empty folder as the root element
//			doc = Document.createNewWithRoot(OperatingSystem.userHome() + "/Desktop/SIPs/Test", OperatingSystem.userHome() + "/Desktop/Test/TopFolder", "sa_all-formats-01", "dss-01", "Document.main");		//	Create a new ZIP Document with an empty folder as the root element

			//	----------	Reading the created SIPs:
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/created/new/mets.xml", "Document.main");						//	Backcheck the previously created Document
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/created/new.zip", "Document.main");							//	Backcheck the previously created Document

			//	----------	Restricted files and folders:
//			doc = Document.openReadWrite("files/RestrictedAccess/mets.xml", "Document.main");								//	METS with files with restricted access
//			doc = Document.openReadWriteFilesLocked("files/RestrictedAccess/mets.xml", "Document.main");					//	METS with files with restricted access
//			Tracer.trace(doc.isAtLeastOneFileNotReadable);
//			Tracer.trace(doc.isAtLeastOneFileNotWritable);
//			for (NodeAbstract n: doc.getStructureMap().getRoot().filesAndFoldersNotReadableByCurrentUser())		Tracer.trace(n);

			//	----------	Dynamic metadata:
//			doc = Document.openReadWrite("files/NestedFolders/mets.xml", "Document.main");									//	METS with Levels
//			NodeAbstract node = doc.getStructureMap().searchId("_20120606105132179");
//			doc = Document.openReadWrite("files/EmptyMets/mets.xml", "Document.main");										//	Empty SIP
//			NodeAbstract node = doc.getStructureMap().getRoot();

//			node.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();

//			node.setDynamicMetadataValueForName("PID", "pid");
//			node.setDynamicMetadataValueForName("unitTitleAdditional", "Additional Title");
//			node.setDynamicMetadataValueForName("accessNr", "Acqckg-Nr");
////			node.setDynamicMetadataValueForName("creationPeriod", "1963-02-09");								//	Default value is specified
//			node.setDynamicMetadataValueForName("creationPeriodNotes", "Starting on the 11th of June 2012");
//			node.setDynamicMetadataValueForName("origination", "Docuteam");
//			node.setDynamicMetadataValueForName("origination", "Docuteam, Dättwil");
//			node.setDynamicMetadataValueForName("origination", "Docuteam, Dättwil, Langacker 16");
////			node.setDynamicMetadataValueForName("origination", null);
////			node.setDynamicMetadataValueForName("title", "Great New Title");
//			node.addDynamicMetadataElementInstanceWithName("language");
//			node.setDynamicMetadataValueForName("language", 0, "DE");
//			node.addDynamicMetadataElementInstanceWithName("language");
//			node.setDynamicMetadataValueForName("language", 1, "EN");
//			node.addDynamicMetadataElementInstanceWithName("language");
//			node.setDynamicMetadataValueForName("language", 2, "FR");
//			node.setDynamicMetadataValueForName("language", 0, "XX");
//			node.setDynamicMetadataValueForName("languageNotes", "Language may be any");
////			node.setDynamicMetadataValueForName("languageNotes", null);
//			node.setDynamicMetadataValueForName("comment", "I think this is really not bad");
//			node.setDynamicMetadataValueForName("extent", "3 mal 3 mal 3 Meter");
//			node.setDynamicMetadataValueForName("material", "Pappe");
//			node.setDynamicMetadataValueForName("refCode", "12345-67890");
//			node.setDynamicMetadataValueForName("refCodeOld", "12345-67890-X");
//			node.setDynamicMetadataValueForName("refCodeAdmin", "12345-67890-A");
//			node.setDynamicMetadataValueForName("scopeContent", "This contains lots of wierd stuff");
//			node.setDynamicMetadataValueForName("accessRestriction", "Access Restriction");
//			node.setDynamicMetadataValueForName("archivalHistory", "Archival History");
//			node.setDynamicMetadataValueForName("conditionsOfReproductions", "Conditions of Reproductions");
//			node.setDynamicMetadataValueForName("reproductions", "Reproductions");
//			node.setDynamicMetadataValueForName("appraisalAndDestruction", "A and D");
//			node.setDynamicMetadataValueForName("arrangement", "Arrangement");
//			node.setDynamicMetadataValueForName("bibliography", "Bib");
//			node.setDynamicMetadataValueForName("descriptionLevelNotes", "description level notes");
//			node.setDynamicMetadataValueForName("descriptionRules", "description rules");
//			node.setDynamicMetadataValueForName("author", "Lem");
//			node.setDynamicMetadataValueForName("institution", "Docuteam, Dättwil");							//	Default value is specified
////			node.setDynamicMetadataValueForName("objectType", "Korrespondenzstück");							//	Default value is specified
//			node.setDynamicMetadataValueForName("revisions", "Revisionen");
////			node.setDynamicMetadataValueForName("retentionPolicy", "Retention policy");							//	Default value is specified
////			node.setDynamicMetadataValueForName("retentionPeriodBaseYear", "2000");								//	Default value is specified
//			node.setDynamicMetadataValueForName("responsible", "Responsible: ich");
//			node.setDynamicMetadataValueForName("involved", "Involved: Egon");
//			node.setDynamicMetadataValueForName("staff", "Staff: Luise");
//			node.setDynamicMetadataValueForName("projectName", "ProjectName");
//			node.setDynamicMetadataValueForName("projectTitle", "ProjectTitle");
//			node.setDynamicMetadataValueForName("project", "Project");
//			node.setDynamicMetadataValueForName("projectAbbreviation", "ProjectAbbr.");
//			node.setDynamicMetadataValueForName("location", "Location");
//			node.setDynamicMetadataValueForName("journal", "Journal");
//			node.setDynamicMetadataValueForName("university", "University");
//			node.setDynamicMetadataValueForName("institute", "Institute");
//			node.setDynamicMetadataValueForName("compartment", "Compartment");
//			node.setDynamicMetadataValueForName("method", "Method");
//			node.setDynamicMetadataValueForName("keyword", "Keyword");
//			node.setDynamicMetadataValueForName("fundingSource", "FundingSource");
//			node.setDynamicMetadataValueForName("status", "Status");
//			node.setDynamicMetadataValueForName("publisher", "Publisher");
//			node.setDynamicMetadataValueForName("doiJournal", "DOI Journal");
//			node.setDynamicMetadataValueForName("retentionPeriod", "Retention Period");
//			node.setDynamicMetadataValueForName("usage", "Use for this and that");
//			node.setDynamicMetadataValueForName("year", "Year 1999");
//			node.setDynamicMetadataValueForName("abstract", "Abstract");

//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstancesWhichAreMandatoryOrSet())		Tracer.trace(mde);
//			node.setLevel(LevelOfDescription.get("Archiv"));		//	Change the level -> some metadata disappear, some are re-initialized

//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstances())							Tracer.trace(mde);
//			Tracer.trace();
//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstancesToBeDisplayed())				Tracer.trace(mde);
//			Tracer.trace();
//			for (MetadataElementInstance mde: node.getDynamicMetadataElementInstancesWhichAreMandatoryButNotSet())	Tracer.trace(mde);

			//	----------	Templates:
//			doc = Document.createNewFromTemplate("files/Templates/Template1", OperatingSystem.userHome() + "/Desktop/Created/SIP1", "Document.main");
//			doc = Document.createNewFromTemplate("files/Templates/Template1", OperatingSystem.userHome() + "/Desktop/Created/SIP2.zip", "Document.main");

//			doc = Document.openReadWrite("files/ch-001194-4_459 Nested", "Document.main");
//			doc = Document.openReadWriteFilesLocked("files/MetsWithMetadata", "Document.main");
//			doc = Document.openReadWriteFilesLocked(OperatingSystem.userHome() + "/Desktop/SIPs/MetsWithMetadata", "Document.main");
//			doc.saveAsTemplate(OperatingSystem.userHome() + "/Desktop/Templates/MetsWithMetadata");

			//	----------	Locking:
//			doc = Document.openReadWrite("files/Locked/mets.xml", "Document.main");
//			doc = Document.openReadWrite("files/ch-001194-4_459 Nested/mets.xml", "Document.main");
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIP/mets.xml", "Document.main");
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIP.zip", "Document.main");
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIP.zip", "Document.main");

			//	----------	Creating EAD File:
//			doc = Document.openReadWriteFilesLocked("files/NestedFolders/mets.xml", "Document.main");
//			doc.createEADFile(OperatingSystem.userHome() + "/Desktop/EAD/NestedFolders.xml");
//			doc = Document.openReadOnly("files/MetsWithMetadata/mets.xml", "Document.main");
//			doc.createEADFile(OperatingSystem.userHome() + "/Desktop/EADs/MetsWithMetadata.xml");

			//	----------	Non-standard levels:
//			Document doc1 = Document.openReadOnly("files/MetsWithMetadata", "Document.main");
//			Document doc2 = Document.openReadOnly("files/MetsWithMetadataNonstandardLevels", "Document.main");
//			Tracer.trace(doc1.getLevels());
//			Tracer.trace(doc2.getLevels());

			//	----------	Submitting:
//			doc = Document.openReadWrite(OperatingSystem.userHome() + "/Desktop/SIPs/DifferentLevelsSubmitRequested", "Document.main");
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setUnitTitle("Test1");
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setSubmitStatus(SubmitStatus.SubmitRequested);
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setUnitTitle("Test2");
//			Tracer.trace(doc.getStructureMap().getRoot().getUnitTitle());
//			doc.getStructureMap().getRoot().setSubmitRequested_recursive();
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());
//			Tracer.trace(doc.submit(OperatingSystem.userHome() + "/Desktop/IngestDropFolder"));
//			Tracer.trace(doc.getStructureMap().getRoot().getSubmitStatus());

			//	----------	Ingesting:
			RosettaConverter.convertTo(OperatingSystem.userHome() + "/Desktop/IngestDropFolder/DifferentLevelsSubmitRequested", OperatingSystem.userHome() + "/Desktop/IEs/DifferentLevelsSubmitRequested", "Document.main");
//			RosettaConverter.convertTo(OperatingSystem.userHome() + "/Desktop/IngestDropFolder/DifferentLevels", OperatingSystem.userHome() + "/Desktop/IEs/DifferentLevels", "Document.main");
//			RosettaConverter.convertTo(OperatingSystem.userHome() + "/Desktop/IngestDropFolder/DifferentFileTypes", OperatingSystem.userHome() + "/Desktop/IEs/DifferentFileTypes", "Document.main");
//			RosettaConverter.convertTo(OperatingSystem.userHome() + "/Desktop/IngestDropFolder/Icons", OperatingSystem.userHome() + "/Desktop/IEs/Icons", "Document.main");
//			RosettaConverter.convertTo(OperatingSystem.userHome() + "/Desktop/IngestDropFolder/METSWithMetadata", OperatingSystem.userHome() + "/Desktop/IEs/METSWithMetadata", "Document.main");

			//	----------	Saving:
//			setBackupFolder(OperatingSystem.userHome() + "/Desktop/Backups");
//			setKeepBackupsCount(3);
//			doc.saveWithBackup();
//			doc.saveWithoutBackup();
//			doc.saveTo(OperatingSystem.userHome() + "/Desktop/Created/Test");
//			doc = Document.openReadOnly(OperatingSystem.userHome() + "/Desktop/Created/Test", "Document.main");		//	Reread saved doc

//			Tracer.trace(doc);				//	ToDo
		}
		catch (Throwable x)
		{
			x.printStackTrace();
		}
		finally
		{
			if (!ExceptionCollector.isEmpty())		Tracer.trace(ExceptionCollector.toStringAll());

			try
			{
				if (doc != null)
				{
					doc.unlockIfNecessary();
					doc.cleanupWorkingCopy();
				}
			}
			catch (java.lang.Exception e)
			{
				e.printStackTrace();
			}
		}
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	----------		Initializing			---------------------------------------------------

	static public void setBackupFolder(String backupFolder)
	{
		BackupFolder = backupFolder;
	}

	/**
	 * Set the keepBackupsCount value. If trying to set to null, ignore it and leave as is.
	 * @param keepBackupsCount Must not be negative.
	 */
	static public void setKeepBackupsCount(Integer keepBackupsCount)
	{
		if (keepBackupsCount == null)		return;
		if (keepBackupsCount < 0)			throw new IllegalArgumentException("KeepBackupsCount must not be negative: " + keepBackupsCount);

		KeepBackupsCount = keepBackupsCount;
	}

	//	--------		Reading				-------------------------------------------------------

	/**
	 * This method is actually a constructor.
	 * It reads the file 'metsFileName' (assuming it is a valid METS file) and generates an instance of Document containing this data.
	 * Since no working copy is created, all actions that manipulate files are disabled.
	 * @throws java.lang.Exception
	 */
	static public Document openReadWriteFilesLocked(String zipOrMETSFilePath, String operatorName) throws java.lang.Exception
	{
		return openReadWriteFilesLocked(zipOrMETSFilePath, operatorName, null);
	}


	/**
	 * This method is actually a constructor.
	 * It reads the file 'metsFileName' (assuming it is a valid METS file) and generates an instance of Document containing this data.
	 * Since no working copy is created, all actions that manipulate files are disabled.
	 * @throws java.lang.Exception
	 */
	static public Document openReadWriteFilesLocked(String zipOrFolderOrMETSFilePath, String operatorName, Observer observer) throws java.lang.Exception
	{
		return readZIPOrFolderOrMETSFile(zipOrFolderOrMETSFilePath, Mode.ReadWriteNoFileOps, operatorName, observer);
	}


	/**
	 * This method is actually a constructor.
	 * It reads the file 'metsFileName' (assuming it is a valid METS file) and generates an instance of Document containing this data.
	 * Since no working copy is created, all actions that manipulate files are disabled.
	 * @throws java.lang.Exception
	 */
	static public Document openReadOnly(String zipOrFolderOrMETSFilePath, String operatorName) throws java.lang.Exception
	{
		return openReadOnly(zipOrFolderOrMETSFilePath, operatorName, null);
	}


	/**
	 * This method is actually a constructor.
	 * Read a SIP in read-only mode, that means that the user can't modify it.
	 * @param zipOrFolderOrMETSFilePath
	 * @param operatorName
	 * @param observer
	 * @return
	 * @throws java.lang.Exception
	 */
	static public Document openReadOnly(String zipOrFolderOrMETSFilePath, String operatorName, Observer observer) throws java.lang.Exception
	{
		return readZIPOrFolderOrMETSFile(zipOrFolderOrMETSFilePath, Mode.ReadOnly, operatorName, observer);
	}


	/**
	 * This method is actually a constructor.
	 * If the metsOrZIPFilePath is a ZIP file, unzip it and open the file 'mets.xml' within.
	 * Otherwise create a working copy of the SIP, then read the file 'metsOrZIPFilePath' (assuming it is a valid METS file)
	 * and generate an instance of Document containing this data.
	 * @throws java.lang.Exception
	 */
	static public Document openReadWrite(String zipOrFolderOrMETSFilePath, String operatorName) throws java.lang.Exception
	{
		return openReadWrite(zipOrFolderOrMETSFilePath, operatorName, null);
	}


	/**
	 * This method is actually a constructor.
	 * If the metsOrZIPFilePath is a ZIP file, unzip it and open the file 'mets.xml' within.
	 * Otherwise create a working copy of the SIP, then read the file 'metsOrZIPFilePath' (assuming it is a valid METS file)
	 * and generate an instance of Document containing this data.
	 * @throws java.lang.Exception
	 */
	static public Document openReadWrite(String zipOrFolderOrMETSFilePath, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (zipOrFolderOrMETSFilePath.toLowerCase().endsWith(".zip"))		return readZIPFile(zipOrFolderOrMETSFilePath, Mode.ReadWrite, operatorName, null);

		File metsFile = new File(zipOrFolderOrMETSFilePath);
		if (!metsFile.exists())							throw new FileNotFoundException(zipOrFolderOrMETSFilePath);
		if (metsFile.isDirectory())						metsFile = new File(zipOrFolderOrMETSFilePath + "/" + DefaultMetsFileName);

		String originalFolder = metsFile.getParent();

		//	If this SIP is already locked by somebody else, don't create a working copy but open it directly (originalSIPFolder is null):
		if (isLockedBySomebodyElse(originalFolder))		return readMETSFile(metsFile.getPath(), null, Mode.ReadWrite, operatorName, observer);

		//	The SIP is a folder - so make a working copy and open this:
		String workingFolder = FileUtil.getTempFolder() + "/" + WorkingFolderPrefix + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

		try
		{
			FileUtil.copyToFolderOverwriting(originalFolder, workingFolder, true);		//	true = Preserve all access rights
		}
		catch (FileUtilExceptionListException ex)
		{
			//	If Exceptions occurred during creating the copy, ignore them and continue!
		}

		String originalFolderName = metsFile.getParentFile().getName();
		String metsFileName = metsFile.getName();

		return readMETSFile(workingFolder + "/" + originalFolderName + "/" + metsFileName, originalFolder, Mode.ReadWrite, operatorName, observer);
	}


	/**
	 * This method is actually a constructor.
	 * If the metsOrZIPFilePath is a ZIP file, unzip it and open the file 'mets.xml' within.
	 * Otherwise create a working copy of the SIP, then read the file 'metsOrZIPFilePath' (assuming it is a valid METS file)
	 * and generate an instance of Document containing this data.
	 * @throws java.lang.Exception
	 */
	static public Document createNewFromTemplate(String templatePath, String zipOrMETSFilePath, String operatorName) throws java.lang.Exception
	{
		return createNewFromTemplate(templatePath, zipOrMETSFilePath, operatorName, null);
	}


	static public Document createNewFromTemplate(String templateFolderPath, String zipOrMETSFilePath, String operatorName, Observer observer) throws java.lang.Exception
	{
		Document document;

		if (zipOrMETSFilePath.toLowerCase().endsWith(".zip"))
		{
			//	It is a ZIP file:
			String sipFolderPath = FileUtil.asFilePathWithoutExtension(zipOrMETSFilePath);

			//	Copy the template to the destination path and zip it:
			FileUtil.copyToOverwriting(templateFolderPath, sipFolderPath, false);
			Zipper.zip(sipFolderPath, zipOrMETSFilePath, true);
			FileUtil.delete(sipFolderPath);

			document = openReadWrite(zipOrMETSFilePath, operatorName, observer);
		}
		else
		{
			//	Copy the template to the destination path (don't preserve the access rights but set all to read/write):
			FileUtil.copyToOverwriting(templateFolderPath, zipOrMETSFilePath, false);

			//	Open the copy:
			document = openReadWrite(zipOrMETSFilePath + "/" + DefaultMetsFileName, operatorName, observer);
		}

		//	Initialize dynamic metadata elements in all nodes from the template:
		for (NodeAbstract node: document.getStructureMap().getRoot().getWithDescendants())	node.initializeDynamicMetadataElementInstancesWhichAreMandatoryOrAlwaysDisplayed();

		document.saveWithoutBackup();
		return document;
	}


	static public Document createNewWithRoot(String newZIPOrFolderFilePath, String sourceFilePath, String saId, String dssId, String operatorName) throws java.lang.Exception
	{
		return createNewWithRoot(newZIPOrFolderFilePath, sourceFilePath, saId, dssId, operatorName, null);
	}


	static public Document createNewWithRoot(String newZIPOrFolderFilePath, String sourceFilePath, String saId, String dssId, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (!new File(sourceFilePath).exists())		throw new FileNotFoundException(sourceFilePath);

		Document document = createEmptySIP(newZIPOrFolderFilePath, saId, dssId, operatorName, observer);
		try
		{
			//	Insert file or folder recursively as the new root in structMap:
			document.structureMap.createRootNode(sourceFilePath);
		}
		catch (FileOperationNotAllowedException ex)
		{
			//	Ignore the FileOperationNotAllowedException because this newly created ZIP will be saved anyway.
			//	Apart from that - this exception will not be thrown anyway because the METS was not opened as 'read-only'.
		}

		//	Save SIP to original folder:
		document.saveWithoutBackup();

		return document;
	}


	static public Document createNewWithRootFolderName(String newZIPOrFolderFilePath, String rootFolderName, String saId, String dssId, String operatorName) throws java.lang.Exception
	{
		return createNewWithRootFolderName(newZIPOrFolderFilePath, rootFolderName, saId, dssId, operatorName, null);
	}


	static public Document createNewWithRootFolderName(String newZIPOrFolderFilePath, String rootFolderName, String saId, String dssId, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (rootFolderName == null || rootFolderName.isEmpty())	throw new FolderNameIsEmptyException();

		Document document = createEmptySIP(newZIPOrFolderFilePath, saId, dssId, operatorName, observer);
		try
		{
			//	Insert new empty folder as the new root in structMap:
			document.structureMap.createNewRootFolder(rootFolderName);
		}
		catch (FileOperationNotAllowedException ex)
		{
			//	Ignore the FileOperationNotAllowedException because this newly created ZIP will be saved anyway.
			//	Apart from that - this exception will not be thrown anyway because the METS was not opened as 'read-only'.
		}

		//	Save SIP to original folder:
		document.saveWithoutBackup();

		return document;
	}

	//	--------		Accessing			-------------------------------------------------------

	static public void setMETS_SIPTemplateFile(String newMETS_SIPTemplateFile)
	{
		METS_SIPTemplateFile = newMETS_SIPTemplateFile;
	}

	static public String getBackupFolder()
	{
		return BackupFolder;
	}

	static public Integer getKeepBackupsCount()
	{
		return KeepBackupsCount;
	}

	//	--------		Inquiring			-------------------------------------------------------

	static public boolean isValidSIPFolder(String folderPath)
	{
		return isValidSIPFolder(new File(folderPath));
	}

	static public boolean isValidSIPFolder(File folder)
	{
		//	Must be an existing and readable folder...
		if (folder == null || !folder.exists() || !folder.canRead())			return false;
		if (folder.isFile())													return false;

		//	... and must contain a mets.xml file:
		File[] metsFilesWithin = folder.listFiles();
		if (metsFilesWithin == null)											return false;
		for (File file: metsFilesWithin)
			if (file.isFile()
				&& file.getName().toLowerCase().equals(DefaultMetsFileName))	return true;

		return false;
	}

	//	--------		Business Ops		-------------------------------------------------------

	/**
	 * Return true if the METS-file and the corresponding SIP appears to be OK; false otherwise.
	 * As a default, regard missing files als errors.
	 *
	 * @param metsFileName contains the file name of the METS file to parse
	 */
	static public boolean check(String metsFileName)
	{
		return check(metsFileName, true);
	}


	/**
	 * Return true if the METS-file and the corresponding SIP appears to be OK; false otherwise.
	 *
	 * @param metsFileName
	 * @param regardMissingFilesAsErrors If true, regard missing files as errors and fill the ExceptionCollector with corresponding exceptions; otherwise don't.
	 * @return
	 */
	static public boolean check(String metsFileName, boolean regardMissingFilesAsErrors)
	{
		Document document = null;

		ExceptionCollector.clear();
		try
		{
			document = readZIPOrFolderOrMETSFile(metsFileName, Mode.ReadOnly, null, null);
		}
		catch (java.lang.Exception e)
		{
			e.printStackTrace();
			return false;
		}

		if (regardMissingFilesAsErrors && document.isAtLeastOneFileNotReadable)
			for (NodeAbstract n: document.getStructureMap().getRoot().getWithDescendants())
			{
				File f = n.getFile();
				if (!f.exists())		Exception.remember(new FileNotFoundException(f.getPath()));
			}

		return ExceptionCollector.isEmpty();
	}

	//	--------		Locking				-------------------------------------------------------

	static public boolean isLocked(File sip)
	{
		return isLocked(sip.getPath());
	}

	/**
	 * Return true if this SIP is locked by somebody else, false otherwise.
	 */
	static public boolean isLocked(String sipName)
	{
		return lockedByWhom(sipName) != null;
	}


	/**
	 * Return true if this SIP is locked by somebody else, false otherwise.
	 */
	static public boolean isLockedBySomebodyElse(File sip)
	{
		return isLockedBySomebodyElse(sip.getPath());
	}

	/**
	 * Return true if this SIP is locked by somebody else, false otherwise.
	 */
	static public boolean isLockedBySomebodyElse(String sipName)
	{
		String lockedBy = lockedByWhom(sipName);

		if (lockedBy == null)		return false;
		return !OperatingSystem.userName().equals(lockedBy);
	}


	/**
	 * Return the login name of the user who locked this SIP, or null if no lock exists.
	 *
	 * @param sipName
	 * @return
	 */
	static public String lockedByWhom(File sip)
	{
		return lockedByWhom(sip.getPath());
	}

	/**
	 * Return the login name of the user who locked this SIP, or null if no lock exists.
	 *
	 * @param sipName
	 * @return
	 */
	static public String lockedByWhom(String sipName)
	{
		File lockFile = new File(sipName + LockFileSuffix);

		//	If no lockfile exists, the SIP is not locked for sure:
		if (!lockFile.exists())		return null;

		//	The lockfile exists. Who locked it?
		return FileUtil.getFileContentAsString(lockFile).trim();		//	trim() is here to cut off any possible newlines.
	}

	//	========	Static Private			=======================================================

	//	--------		Initialization		-------------------------------------------------------

	static private Document createEmptySIP(String newZIPOrFolderFilePath, String saId, String dssId, String operatorName, Observer observer) throws java.lang.Exception
	{
		File newZIPOrFolderFile = new File(newZIPOrFolderFilePath);
		if (newZIPOrFolderFile.exists())
		{
			if (newZIPOrFolderFile.isFile())				FileUtil.delete(newZIPOrFolderFile);
			else if (newZIPOrFolderFile.isDirectory()
				&& newZIPOrFolderFile.list().length != 0)	throw new FolderIsNotEmptyException(newZIPOrFolderFilePath);
		}

		newZIPOrFolderFile.getParentFile().mkdirs();

		String newSIPFolderName = new File(newZIPOrFolderFilePath).getName();
		//	Cut off ".zip" or ".ZIP" from the file name if necessary to get the SIP folder name:
		if (FileUtil.asFileNameExtension(newSIPFolderName).toLowerCase().equals("zip"))
			newSIPFolderName = FileUtil.asFileNameWithoutExtension(newSIPFolderName);

		String workingFolder = FileUtil.getTempFolder() + "/" + WorkingFolderPrefixCreateZip + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);
		FileUtil.delete(workingFolder);
		String newFolderPath = workingFolder + "/" + newSIPFolderName;

		//	Copy METS template to temporary destination folder:
		String newMETSFileName = newFolderPath + "/" + DefaultMetsFileName;
		FileUtil.copyToOverwriting(METS_SIPTemplateFile, newMETSFileName, false);

		//	Read the newly created METS file:
		Document document = readMETSFile(newMETSFileName, newZIPOrFolderFilePath, Mode.ReadWrite, operatorName, observer);
		document.saId = saId;
		document.dssId = dssId;
		document.setType(saId + "_" + dssId);

		return document;
	}

	//	--------		Reading				-------------------------------------------------------

	/**
	 * This method is actually a constructor. It reads the ZIP or folder or METS file 'zipOrFolderOrMETSFilePath'
	 * and generates an instance of Document containing the data.
	 * If it is a folder, assume that the DefaultMetsFileName is within and open this.
	 * If 'originalSipFolder' is null, no working copy is created and all actions that manipulate files are disabled.
	 * If 'zipOrMETSFilePath' is a zip file, a working copy is created anyway.
	 *
	 * @param zipOrFolderOrMETSFilePath contains the file name of the METS file to parse, or the name of the ZIP file, or the folder containing the METS file.
	 * @param originalSipFolder may be null - in this case we are working on the original SIP, NOT on a working copy, and operations manipulating files are not allowed.
	 * @param operatorName an identification string of the caller of this method. Any manipulations are tagged as performed by this operatorName.
	 * @return The generated document
	 * @throws java.lang.Exception
	 */
	static private Document readZIPOrFolderOrMETSFile(String zipOrFolderOrMETSFilePath, Mode mode, String operatorName, Observer observer) throws java.lang.Exception
	{
		return
			(zipOrFolderOrMETSFilePath.toLowerCase().endsWith(".zip"))
				? readZIPFile(zipOrFolderOrMETSFilePath, mode, operatorName, observer)
				: (new File(zipOrFolderOrMETSFilePath).isFile())
					? readMETSFile(zipOrFolderOrMETSFilePath, null, mode, operatorName, observer)
					: readMETSFile(zipOrFolderOrMETSFilePath + "/" + DefaultMetsFileName, null, mode, operatorName, observer);
	}


	/**
	 * This method is actually a constructor.
	 * It extracts the ZIP file into the working folder, then reads the METS file and generates an instance of Document containing the data.
	 *
	 * @param zipFileName contains the file name of the ZIP file to parse
	 * @throws java.lang.Exception
	 */
	static private Document readZIPFile(String zipFileName, Mode mode, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (!new File(zipFileName).exists())		throw new FileNotFoundException(zipFileName);

		//	Unzip into working folder:
		String sipName = (new File(zipFileName).getName());
		sipName = sipName.substring(0, sipName.length() - 4);
		String workingFolder = FileUtil.getTempFolder() + "/" + WorkingFolderPrefixReadZip + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "/" + sipName;

		Zipper.unzip(zipFileName, workingFolder);

		if (!new File(workingFolder + "/" + DefaultMetsFileName).exists())
		{
			try
			{
				FileUtil.delete(new File(workingFolder).getParent());
			}
			catch (FileUtilExceptionListException ex)
			{
				ex.printStackTrace();
			}
			throw new FileNotFoundException("'" + DefaultMetsFileName + "' not found in zip-file '" + zipFileName + "'");
		}

		return readMETSFile(workingFolder + "/" + DefaultMetsFileName, zipFileName, mode, operatorName, observer);
	}


	/**
	 * This method is actually a constructor. It reads the METS file 'metsFilePath' (assuming it is a valid METS file)
	 * and generates an instance of Document containing the data.
	 * If 'originalSipFolder' is null, no working copy is created and all actions that manipulate files are disabled.
	 *
	 * @param metsFilePath contains the file name of the METS file to parse
	 * @param originalSIPFolder contains the original folder path or a ZIP file path. It may be null - in this case we are working on the original SIP, NOT on a working copy, and operations manipulating files are not allowed.
	 * @param operatorName an identification string of the caller of this method. Any manipulations are tagged as performed by this operatorName.
	 * @return The generated document
	 * @throws java.lang.Exception
	 */
	static private Document readMETSFile(String metsFilePath, String originalSIPFolder, Mode mode, String operatorName, Observer observer) throws java.lang.Exception
	{
		if (!new File(metsFilePath).exists())		throw new FileNotFoundException(metsFilePath);

		Document document;
		File metsFile = new File(metsFilePath);

		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new DocumentFactory());
		document = (Document)reader.read(metsFile);

		document.initialize(metsFile, originalSIPFolder, mode, operatorName, observer);

		try
		{
			return document.parse();
		}
		catch (java.lang.Exception ex)
		{
			//	Cleanup if parsing was not successful:
			document.unlockIfNecessary();
			document.cleanupWorkingCopy();

			//	Re-throw the exception so it doesn't get lost:
			throw ex;
		}
	}

	//	========	Instance Public			=======================================================

	//	----------		Initializing			---------------------------------------------------
	//	----------		Accessing				---------------------------------------------------

	public String getNamespaceMETS()
	{
		return this.namespaceMETS;
	}

	public String getNamespacePREMIS()
	{
		return this.namespacePREMIS;
	}

	public Mode getMode()
	{
		return this.mode;
	}

	public String getProfile()
	{
		return this.profile;
	}

	public String getLabel()
	{
		return this.label;
	}

	public String getType()
	{
		return this.type;
	}

	public void setType(String type)
	{
		this.type = type;
		this.getRootElement().addAttribute("TYPE", type);
		this.setIsModified();
	}

	public String getSAId()
	{
		return this.saId;
	}

	public String getDSSId()
	{
		return this.dssId;
	}

	public String getObjectId()
	{
		return this.objId;
	}

	public void setObjectId(String objId)
	{
		this.objId = objId;
		this.getRootElement().addAttribute("OBJID", objId);
		this.setIsModified();
	}

	public String getSipFolder()
	{
		return new File(this.filePath).getParent();
	}

	public String getOriginalSipFolder()
	{
		return this.originalSipFolder;
	}

	public String getOriginalSipName()
	{
		return new File(this.originalSipFolder).getName();
	}


	public Header getHeader()
	{
		return this.header;
	}

	public AMDSection getAMDSection()
	{
		return this.amdSection;
	}

	public Collection<DMDSectionAbstract> getDMDSections()
	{
		return this.dmdSections.values();
	}

	public DMDSectionAbstract getDMDSection(String id)
	{
		return this.dmdSections.get(id);
	}

	public void addDMDSection(DMDSectionAbstract dmdSection)
	{
		this.dmdSections.put(dmdSection.getId(), dmdSection);
		this.setIsModified();
	}

	public FileSection getFileSection()
	{
		return this.fileSection;
	}

	public StructureMap getStructureMap()
	{
		return this.structureMap;
	}


	public String getCurrentOperatorName()
	{
		return this.currentOperatorName;
	}

	public void setCurrentOperatorName(String opName)
	{
		this.currentOperatorName = opName;
		this.setIsModified();
	}


	public List<NodeFile> setSubmissionAgreement( String saId, String dssId) throws FileNotFoundException, DocumentException, ExceptionCollectorException
	{
		SubmissionAgreement submissionAgreement = SubmissionAgreement.read(saId + ".xml");
		if (submissionAgreement == null)		return null;

		this.saId = submissionAgreement.getSaId();
		this.dssId = dssId;
		this.setType(this.saId + "_" + dssId);
		this.submissionAgreement = submissionAgreement;

		return this.filesNotAllowedBySubmissionAgreement();
	}


	public SubmissionAgreement getSubmissionAgreement()
	{
		if (this.saId == null || this.saId.isEmpty())		return null;

		try
		{
			//	Cache the submissionAgreement once it is requested:
			if (this.submissionAgreement == null)		this.submissionAgreement = SubmissionAgreement.read(this.saId + ".xml");
			return this.submissionAgreement;
		}
		catch (java.lang.Exception e)
		{
			e.printStackTrace();
			return null;
		}
	}


	public String getLockedBy()
	{
		return this.lockedBy;
	}

	//	--------		Inquiring			-------------------------------------------------------

	@Override
	public boolean isReadOnly()
	{
		return this.mode == Mode.ReadOnly;
	}

	public boolean isLocked()
	{
		return this.mode == Mode.Locked;
	}

	public boolean isReadWrite()
	{
		return this.mode == Mode.ReadWrite;
	}

	public boolean isReadWriteNoFileOps()
	{
		return this.mode == Mode.ReadWriteNoFileOps;
	}

	public boolean canWrite()
	{
		return this.mode == Mode.ReadWrite || this.mode == Mode.ReadWriteNoFileOps;
	}

	public boolean areFileOperationsAllowed()
	{
		return this.mode == Mode.ReadWrite;
	}

	public boolean hasNodesWithDynamicMetadataElementInstancesWhichAreMandatoryButNotSet()
	{
		return this.structureMap.hasNodesWithDynamicMetadataElementInstancesWhichAreMandatoryButNotSet();
	}


	public boolean checkFixity()
	{
		return this.structureMap.checkFixity();
	}


	public boolean checkAgainstSubmissionAgreement()
	{
		return this.structureMap.checkAgainstSubmissionAgreement();
	}

	public List<NodeFile> filesNotAllowedBySubmissionAgreement()
	{
		return this.structureMap.filesNotAllowedBySubmissionAgreement();
	}


	public boolean checkAgainstSubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return this.structureMap.checkAgainstSubmissionAgreement(sa, dssId);
	}

	public List<NodeFile> filesNotAllowedBySubmissionAgreement(SubmissionAgreement sa, String dssId)
	{
		return this.structureMap.filesNotAllowedBySubmissionAgreement(sa, dssId);
	}

	//	--------		Interface			-------------------------------------------------------

	/**
	 * Return the type of this document: METS, EAD, ...
	 */
	@Override
	public Type getDocumentType()
	{
		return Type.METS;
	}

	//	--------		Business Ops		-------------------------------------------------------

	public void createEADFile(String eadFilePath) throws DocumentException, ExceptionCollectorException, IOException
	{
		ch.docuteam.docudarc.ead.Document eadDocument = ch.docuteam.docudarc.ead.Document.createEmpty();
		eadDocument.setId(this.getType());

		//	Since I know that the archDesc already exists (initialized with default values), I have only to initialize it explicitly:
		eadDocument.getArchDesc().initialize(this.structureMap.getRoot().getUnitTitle());
		eadDocument.getArchDesc().fillEADElementsFromNodeRecursively(this.structureMap.getRoot());

		eadDocument.saveAs(eadFilePath);
	}

	//	--------		Submitting			-------------------------------------------------------

	/**
	 * Set all submit stati that are "SubmitRequested" to "SubmitRequestPending", then trigger the feeder-workflow "Ingest".
	 * If I am readOnly or locked, reject.
	 * If I don't contain any files that have the submit status "SubmitRequest", reject because there is nothing marked to be ingested.
	 * If I contain files that have the submit status "SubmitRequestPending", reject because I am already being ingested.
	 * @return a String denoting whether the ingest workflow was started or if some problems occured.
	 * @throws java.lang.Exception
	 */
	public String submit(String submitFolder) throws java.lang.Exception
	{
		//	Is this document read-only or locked? If yes, reject:
		if (this.isReadOnly())				return "MessageSubmitSIPIsReadOnly";
		if (this.isLocked())				return "MessageSubmitSIPIsLocked";

		//	Are there any nodes with SubmitStatus = "SubmitRequestPending"? If yes, reject (SIP is currently in an ongoing submission process):
		//	Are there any nodes with SubmitStatus = "SubmitRequested"? If no, reject (There are no nodes to be submitted):
		boolean hasNodesWithSubmitRequested = false;
		for (NodeAbstract n: this.structureMap.getRoot().getWithDescendants())
		{
			SubmitStatus submitStatus = n.getSubmitStatus();
			if (submitStatus == SubmitStatus.SubmitRequestPending)		return "MessageSubmitSIPIsAlreadySubmitPending";
			if (submitStatus == SubmitStatus.SubmitRequested)			hasNodesWithSubmitRequested = true;
		}
		if (!hasNodesWithSubmitRequested)	return "MessageSubmitSIPHasNoSubmitRequests";

		//	Does this SIP already exist in the submit folder? If yes, reject:
		if (new File(submitFolder + "/" + this.getOriginalSipName()).exists())		return "MessageSubmitSIPExistsAlreadyInSubmitFolder";

		//	Set all nodes with SubmitStatus = "SubmitRequested" to "SubmitRequestPending":
		this.structureMap.setSubmitRequestPending();

		//	Save it:
		this.saveWithBackup();

		//	Lock it as "SubmitRequestPending":
		this.lockAsSubmitRequestPending();

		//	ToDo: Trigger feeder-workflow "Ingest" with either a copy of this SIP or its URL:
		FileUtil.copyToFolderOverwriting(this.originalSipFolder, submitFolder);

		return "MessageSubmitDone";
	}

	//	--------		Persistence			-------------------------------------------------------

	/**
	 * Save the METS-file and create a backup of the original.
	 * If this document has at least one file which is not readable, save without backup!
	 * If this document has at least one file which is not writable, save with backup but ignore FileUtilExceptionListException!
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 * @throws OriginalSIPIsMissingException The original SIP could not be found so no backup could be made. Nevertheless the saving was successful.
	 */
	public void saveWithBackup() throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException, OriginalSIPIsMissingException
	{
		if (this.isAtLeastOneFileNotReadable)
			//	If some files or folders are not readable, I can't make a backup!
			try
			{
				this.save(false);
			}
			catch (FileUtilExceptionListException ex)
			{
				//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
			}
		else if (this.isAtLeastOneFileNotWritable)
			try
			{
				this.save(true);
			}
			catch (FileUtilExceptionListException ex)
			{
				//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
			}
		else
			this.save(true);
	}


	/**
	 * Save the METS-file overwriting the original.
	 * If this document has at least one file which is not readable, ignore FileUtilExceptionListException!
	 * If this document has at least one file which is not writable, ignore FileUtilExceptionListException!
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 */
	public void saveWithoutBackup() throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException
	{
		try
		{
			if (this.isAtLeastOneFileNotReadable)
				//	If some files or folders are not readable, I can't make a backup!
				try
				{
					this.save(false);
				}
				catch (FileUtilExceptionListException ex)
				{
					//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
				}
			else if (this.isAtLeastOneFileNotWritable)
				try
				{
					this.save(false);
				}
				catch (FileUtilExceptionListException ex)
				{
					//	Ignore FileUtilExceptionListException because it's OK that some files or folders couldn't be copied.
				}
			else
				this.save(false);
		}
		catch (OriginalSIPIsMissingException ex)
		{
			//	This exception gets only thrown when saving WITH backup, so it will never happen here
		}
	}


	/**
	 * Save the SIP and rename the METS-file.
	 * This operation is allowed even when this document was opened in read-only mode.
	 * @param newMETSFilePath
	 * @throws DocumentIsReadOnlyException
	 * @throws FileUtilExceptionListException
	 * @throws IOException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 */
	public void saveAs(String newMETSFilePath) throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException
	{
		if (newMETSFilePath.toLowerCase().endsWith(".xml"))
		{
			//	The mets.xml file gets a new name:

			String oldMETSFileName = FileUtil.asFileName(this.filePath);
			String newSIPFolderPath = FileUtil.asParentPath(newMETSFilePath);
			String newMETSFileName = FileUtil.asFileName(newMETSFilePath);

			//	Save to folder:
			this.saveTo(newSIPFolderPath);

			//	Rename the mets.xml file (if isWorkingCopy, then this is the working copy, otherwise this is the original):
			File newFilePath = FileUtil.renameTo(this.filePath, this.getSipFolder() + "/" + newMETSFileName);
			this.filePath = newFilePath.getPath();

			if (this.isWorkingCopy)
			{
				//	If working on a copy, rename the ORIGINAL copy of the mets.xml file too:
				FileUtil.renameTo(newSIPFolderPath + "/" + oldMETSFileName, newMETSFilePath);
			}
		}
		else
		{
			this.saveTo(newMETSFilePath);
		}
	}


	/**
	 * Save the SIP into the given directory.
	 * This operation is allowed even when this document was opened in read-only mode.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException
	 * @throws FileOrFolderIsInUseException  A file or folder inside this SIP is being used by another application.
	 */
	public void saveTo(String newSipFolder) throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException
	{
		boolean wasWorkingCopy = this.isWorkingCopy;

		try
		{
			//	Set isWorkingCopy to false, so nothing gets written to the original SIP:
			this.isWorkingCopy = false;

			//	Save METS-File before copying to new destination (working copy only), but only if it was modified:
			if (this.isModified)	this.saveWithoutBackup();

			//	Copy to new destination:
			if (newSipFolder.toLowerCase().endsWith(".zip"))
			{
				//	Create a zip file:
				List<String> files = new ArrayList<String>();
				for (File file: new File(this.getSipFolder()).listFiles())		files.add(file.getPath());

				Zipper.zip(files.toArray(new String[]{}), newSipFolder);
			}
			else
			{
				//	Copy the SIP, don't preserve the access rights but set all to read/write:
				FileUtil.copyToOverwriting(this.getSipFolder(), newSipFolder, false);
			}
		}
		finally
		{
			this.unlockIfNecessary();

			if (wasWorkingCopy)
			{
				//	This was a working copy, so set the original folder to the one just created;
				//	the working folder (= file path) remains as it was:
				this.originalSipFolder = newSipFolder;
				this.isWorkingCopy = true;
			}
			else
			{
				//	This was NOT a working copy, so set my working folder (= file path) to the path just created
				//	and the original folder to its parent folder.
				File newFile = new File(FileUtil.asCanonicalFileName(newSipFolder + "/" + new File(this.filePath).getName()));
				this.filePath = newFile.getPath();
				this.originalSipFolder = newFile.getParent();
			}

			this.mode = Mode.ReadWrite;
			this.lockIfPossibleAndRequired();
		}
	}


	public void saveAsTemplate(String templatePath) throws java.lang.Exception
	{
		//	Can't create a template when the root node is a file:
		if (this.structureMap.getRoot().isFile())		throw new CantCreateTemplateWithRootFileException();

		//	First save the document if necessary:
		if (this.isModified)		this.saveWithoutBackup();

		//	Then create a copy of this document in the template folder with the given template name (templatePath):
		if (this.isWorkingCopy)
		{
			if (this.isZIPFile())
			{
				Zipper.unzip(this.originalSipFolder, templatePath);
			}
			else
			{
				FileUtil.copyToOverwriting(this.originalSipFolder, templatePath, false);
			}
		}
		else
		{
			FileUtil.copyToOverwriting(new File(this.filePath).getParent(), templatePath, false);
		}

		//	Read the just created template:
		Document templateDocument = openReadWrite(templatePath + "/" + DefaultMetsFileName, "Document.saveAsTemplate");

		//	Then strip away all files:
		for (NodeFile f: ((NodeFolder)templateDocument.getStructureMap().getRoot()).getDescendantFiles())
		{
			try
			{
				f.delete();
				f.getMyDigiprov().delete();
			}
			catch (FileOperationNotAllowedException ex){}		//	This can't happen because I opened the document in read/write mode.
		}

		//	Then strip away all dynamic metadata:
		for (NodeAbstract n: ((NodeFolder)templateDocument.getStructureMap().getRoot()).getWithDescendants())
		{
			for (MetadataElementInstance mdei: n.getDynamicMetadataElementInstances())
			{
				//	Don't remove element if marked with "keepInTemplate='true'":
				if (mdei.keepInTemplate())		continue;

				if (mdei.isRepeatable())
				{
					try
					{
						//	No matter what index THIS metadata instance has, delete always the 0th index.
						//	This way all repeating instances of this metadata will be deleted:
						n.deleteDynamicMetadataElementInstanceWithName(mdei.getName(), 0);
					}
					catch (MetadataElementCantDeleteException ex)
					{
						//	If I can't delete this element, I clear it:
						n.setDynamicMetadataValueForName(mdei.getName(), 0, null);
					}
				}
				else
				{
					try
					{
						//	Delete this element:
						n.deleteDynamicMetadataElementInstanceWithName(mdei.getName(), mdei.getIndex());
					}
					catch (MetadataElementCantDeleteException ex)
					{
						//	If I can't delete this element, I clear it:
						n.setDynamicMetadataValueForName(mdei.getName(), mdei.getIndex(), null);
					}
				}
			}
		}

		//	Finally save the new template:
		templateDocument.saveWithoutBackup();

		//	... and cleanup:
		templateDocument.unlockIfNecessary();
		templateDocument.cleanupWorkingCopy();
	}


	/**
	 * Remove the working copy directory. If no working copy was created, ignore it.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 */
	public void cleanupWorkingCopy() throws IOException, FileUtilExceptionListException
	{
		//	The following applies only if the document was opened using "createWorkingCopyAndRead()",
		//	OR if the source was a zip-file:
		if (this.isWorkingCopy)
		{
			//	Set cleanup folder recursively to writable (if possible) so cleanup can be done:
			FileUtil.setWritable(new File(this.filePath).getParentFile().getParentFile());

			//	Cleanup working folder:
			FileUtil.deleteOnExit(new File(this.filePath).getParentFile().getParentFile());
		}
	}

	//	--------		Locking				-------------------------------------------------------

	/**
	 * If it was me who locked this archive, unlock it.
	 * This method has to be public because it may be called on shutdown or when exceptions occur.
	 */
	public void unlockIfNecessary()
	{
		if (this.lockedBy == null)		this.unlock();
	}

	//	--------		Searching			-------------------------------------------------------

	/**
	 * Search in the metadata of all nodes for the searchString
	 * @param searchString
	 * @return the list of hits
	 */
	public List<NodeAbstract>searchFor(String searchString)
	{
		List<NodeAbstract> searchHits = new Vector<NodeAbstract>();
		for (NodeAbstract node: this.structureMap.getRoot().getWithDescendants())	if (node.searchFor(searchString))	searchHits.add(node);

		return searchHits;
	}

	//	--------		Support				-------------------------------------------------------

	public void deleteDMDSection(DMDSectionAbstract dmdSection)
	{
		this.dmdSections.remove(dmdSection.getId());
	}

	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n====\tMETS-Document on '")
			.append(this.filePath)
			.append("'\t====\n====\tOriginal SIP:    '")
			.append(this.originalSipFolder)
			.append("'\t====\n\t")
			.append(this.label)
			.append("/")
			.append(this.type)
			.append("/")
			.append(this.objId)
			.append(this.isAtLeastOneFileNotReadable? " (not all readable)": " (all readable)")
			.append(this.isAtLeastOneFileNotWritable? " (not all writable)": " (all writable)")
			.append(" (")
			.append(this.mode)
			.append(")")
			.append(this.isWorkingCopy? " (workingCopy)": " (direct)")
			.append(this.lockedBy == null? "": " (locked by: '" + this.lockedBy + "')")
			.append(this.header);

		for (DMDSectionAbstract dmdSection: this.dmdSections.values())		buf.append(dmdSection.toString());

		buf.append(this.amdSection)
			.append(this.fileSection)
			.append(this.structureMap)
			.append("\n");

		return buf.toString();
	}

	//	----------		Temporary				---------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Initialize my variables filePath, originalSipFolder, isWorkingCopy, mode, and currentOperatorName.
	 * Lock the file if necessary. Add the observer to my observer list.
	 */
	protected void initialize(File metsFile, String originalSipFolder, Mode mode, String operatorName, Observer observer)
	{
		this.mode = mode;
		this.filePath = FileUtil.asCanonicalFileName(metsFile);
		this.currentOperatorName = operatorName;
		this.addObserver(observer);

		//	originalSipFolder is either a folder path, or a ZIP file path, or null.
		//	If originalSipFolder is null, I am working on the original SIP (no working copy),
		//	so set my originalSipFolder to the parent of the metsFilePath, and set the property isWorkingCopy to false:
		if (originalSipFolder == null)
		{
			this.originalSipFolder = metsFile.getParent();
			this.isWorkingCopy = false;
		}
		else
		{
			this.originalSipFolder = originalSipFolder;
			this.isWorkingCopy = true;
		}

		//	Locking:
		this.lockIfPossibleAndRequired();
	}

	/**
	 * This method initializes the properties fileSection and structureMap from the data within the METS-file and from the
	 * files and folders referenced by the METS-file.
	 */
	protected Document parse()
	{
		Element root = this.getRootElement();
		this.namespaceMETS = root.getNamespaceForPrefix("METS").getStringValue();
		Namespace premis = root.getNamespaceForPrefix("PREMIS");
		this.namespacePREMIS = (premis != null)? premis.getStringValue(): "info:lc/xmlns/premis-v2";
		this.profile = root.attributeValue("PROFILE");
		this.label = root.attributeValue("LABEL");
		this.type = root.attributeValue("TYPE");
		if ((this.type != null) && (this.type.lastIndexOf("_") != -1))
		{
			this.saId = this.type.substring(0, this.type.lastIndexOf("_"));
			this.dssId = this.type.substring(this.type.lastIndexOf("_") + 1, this.type.length());
		}
		this.objId = root.attributeValue("OBJID");

		//	Optional elements:
		this.distributeMessage("DMD Section...");
		this.dmdSections = DMDSectionAbstract.parse(this);

		//	Mandatory elements:
		this.distributeMessage("Header...");
		this.header = Header.parse(this);
		this.distributeMessage("AMD Section...");
		this.amdSection = AMDSection.parse(this);
		this.distributeMessage("File Section...");
		this.fileSection = FileSection.parse(this);
		this.distributeMessage("Struct Map...");
		this.structureMap = StructureMap.parse(this);
		this.distributeMessage("OK");

		this.isModified = false;

		return this;
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------

	protected boolean isZIPFile()
	{
		return this.originalSipFolder.toLowerCase().endsWith(".zip");
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------

	/**
	 * Write back to original SIP-folder or ZIP-file.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws DocumentIsReadOnlyException This Document can't be saved because it it read-only.
	 * @throws FileOrFolderIsInUseException A file or folder inside this SIP is being used by another application.
	 * @throws OriginalSIPIsMissingException The original SIP could not be found so no backup could be made. Nevertheless the saving was successful.
	 */
	protected void save(boolean withBackup) throws IOException, FileUtilExceptionListException, DocumentIsReadOnlyException, FileOrFolderIsInUseException, OriginalSIPIsMissingException
	{
		//	Can't save if this document is read-only:
		if (!this.canWrite())		throw new DocumentIsReadOnlyException();

		//	If the original SIP is missing, remember this and throw an exception after saving.
		boolean originalSIPIsMissing = false;
		this.header.setLastModificationDateToNow();

		if (this.isWorkingCopy)
		{
			//	Write the mets-file in working copy:
			this.writeMETSFile();

			if (this.isZIPFile())
			{
				//	Original SIP is a zip file:

				if (withBackup)
				{
					if (!new File(this.originalSipFolder).exists())
					{
						originalSIPIsMissing = true;
					}
					else
					{
						//	Backup original ZIP-file:
						if (BackupFolder != null)
						{
							FileUtil.renameTo(this.originalSipFolder, BackupFolder + "/" + FileUtil.asFileNameWithoutExtension(this.originalSipFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + FileUtil.asFileNameExtension(this.originalSipFolder));
						}
						else
						{
							FileUtil.renameTo(this.originalSipFolder, FileUtil.asFilePathWithoutExtension(this.originalSipFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + "." + FileUtil.asFileNameExtension(this.originalSipFolder));
						}

						this.purgeBackups();
					}
				}

				//	Zip working folder back to original ZIP-file:
				List<String> files = new ArrayList<String>();
				for (File file: new File(this.getSipFolder()).listFiles())		files.add(file.getPath());

				Zipper.zip(files.toArray(new String[]{}), this.originalSipFolder);
			}
			else
			{
				//	Original SIP is a folder:

				if (withBackup)
				{
					if (!new File(this.originalSipFolder).exists())
					{
						originalSIPIsMissing = true;
					}
					else
					{
						//	Backup original folder, DO preserve the access rights:
						if (BackupFolder != null)
						{
							FileUtil.copyToOverwriting(this.originalSipFolder, BackupFolder + "/" + FileUtil.asFileName(this.originalSipFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
						}
						else
						{
							FileUtil.copyToOverwriting(this.originalSipFolder, this.originalSipFolder + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
						}

						this.purgeBackups();
					}
				}

				//	Redesign: instead of just copying the working copy back over the original SIP, now copy the working copy to a temporary SIP,
				//	then delete the original SIP and finally rename the temporary SIP to the original SIP name.
				//	So I can be sure there is a saved copy in place.
				String securitySuffix = "_COPY_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);

				//	If there are access-restricted files, I might not be able to delete them - so I overwrite whatever I can:
				if (this.isAtLeastOneFileNotReadable || this.isAtLeastOneFileNotWritable)
				{
					//	There are access-restricted files in this SIP: report but ignore exceptions:

					try
					{
						//	Create security copy:
						FileUtil.copyToOverwriting(this.getSipFolder(), this.originalSipFolder + securitySuffix, true);
					}
					catch (FileUtilExceptionListException ex)
					{
						//	If I have at least one access-restricted file, report but ignore this exception:
						ex.printStackTrace();
					}

					try
					{
						//	Copy securirty copy to original:
						FileUtil.copyToOverwriting(this.originalSipFolder + securitySuffix, this.originalSipFolder, true);
					}
					catch (FileUtilExceptionListException ex)
					{
						//	If I have at least one access-restricted file, report but ignore this exception:
						ex.printStackTrace();
					}

					//	Delete security copy:
					FileUtil.setWritable(this.originalSipFolder + securitySuffix);
					FileUtil.delete(this.originalSipFolder + securitySuffix);
				}
				else
				{
					//	There are supposed to be NO access-restricted files in this SIP, so don't catch exceptions:

					//	Create security copy:
					FileUtil.copyToOverwriting(this.getSipFolder(), this.originalSipFolder + securitySuffix, true);

					try
					{
						//	Delete original:
						FileUtil.delete(this.originalSipFolder);
					}
					catch (FileUtilExceptionListException ex)
					{
						List<FileUtilException> exFiles = new Vector<FileUtilException>();
						for (FileUtilException x: ex.getAllExceptions())		exFiles.add(x);

						//	A file must be in use by another application or the Windows explorer:
						throw new FileOrFolderIsInUseException(exFiles, this.originalSipFolder, this.originalSipFolder + securitySuffix);
					}

					//	Rename security copy back to original:
					FileUtil.renameTo(this.originalSipFolder + securitySuffix, this.originalSipFolder);
				}
			}
		}
		else
		{
			//	No working copy:

			if (withBackup)
			{
				if (!new File(this.originalSipFolder).exists())
				{
					originalSIPIsMissing = true;
				}
				else
				{
					//	Backup original folder, DO preserve the access rights:
					if (BackupFolder != null)
					{
						FileUtil.copyToOverwriting(this.originalSipFolder, BackupFolder + "/" + FileUtil.asFileName(this.originalSipFolder) + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
					}
					else
					{
						FileUtil.copyToOverwriting(this.originalSipFolder, this.originalSipFolder + "_ORIGINAL_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs), true);
					}

					this.purgeBackups();
				}
			}

			//	In the case that no working copy was created, write back only the METS-File:
			this.writeMETSFile();
		}

		this.isModified = false;

		if (originalSIPIsMissing)		throw new OriginalSIPIsMissingException(this.originalSipFolder);
	}


	private void writeMETSFile() throws IOException
	{
		Writer oswriter = new java.io.OutputStreamWriter(new FileOutputStream(this.filePath), "utf-8");
		OutputFormat.createPrettyPrint();
		OutputFormat outformat = new OutputFormat("	", false, "utf-8");
		XMLWriter writer = new XMLWriter(oswriter, outformat);
		writer.write(this);
		writer.flush();
		writer.close();
	}


	private void purgeBackups()
	{
		//	If KeepBackupsCount is null, keep all backups:
		if (KeepBackupsCount == null)		return;

		//	Define the correct filename filter:
		final String backupPattern =
			this.isZIPFile()
				? FileUtil.asFileNameWithoutExtension(this.originalSipFolder) + "_ORIGINAL_"
				: FileUtil.asFileName(this.originalSipFolder) + "_ORIGINAL_";
		FilenameFilter myBackups = new FilenameFilter()
		{	@Override public boolean accept(File dir, String name) { return name.startsWith(backupPattern); }};

		//	Apply the filename filter to the correct backup folder and sort the result according to the file name:
		File backupFolder = BackupFolder != null? new File(BackupFolder): new File(this.originalSipFolder).getParentFile();
		List<String> backups = Arrays.asList(backupFolder.list(myBackups));
		Collections.sort(backups);

		//	Purge all superflouos SIPs:
		for (int i = 0; i < backups.size() - KeepBackupsCount; i++)
		{
			Logger.info("Purging '" + backupFolder + "/" + backups.get(i) + "'");
			try
			{
				FileUtil.delete(backupFolder + "/" + backups.get(i));
			}
			catch (FileUtilExceptionListException ex)
			{
				ex.printStackTrace();
			}
		}
	}

	//	--------		Locking				-------------------------------------------------------

	/**
	 * If this document is not locked by somebody else, lock it:
	 */
	protected void lockIfPossibleAndRequired()
	{
		String lockedBy = this.lockedByWhom();
		if (lockedBy == null)
		{
			//	Not yet locked:
			this.lockedBy = null;
			if (this.canWrite())		this.lock();
		}
		else if (OperatingSystem.userName().equals(lockedBy))
		{
			//	Is already locked by me:
			this.lockedBy = null;
		}
		else
		{
			//	Is locked by somebody else:
			this.lockedBy = lockedBy;
			this.mode = Mode.Locked;
		}
	}

	/**
	 * Return the login name of the user who locked this SIP, or null if no lock exists.
	 *
	 * @param sipName
	 * @return
	 */
	protected String lockedByWhom()
	{
		return lockedByWhom(this.originalSipFolder);
	}

	/**
	 * Lock this SIP by creating a lock file.
	 */
	protected void lock()
	{
		try
		{
			FileUtil.createFileWithContent(this.originalSipFolder + LockFileSuffix, OperatingSystem.userName());
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
		}
	}

	/**
	 * Unlock this SIP by deleting the lock file (only if allowed).
	 * return true if unlock was succesful, false otherwise.
	 */
	protected boolean unlock()
	{
		//	Can't unlock if I am not allowed to write.
		if (!this.canWrite())		return false;

		try
		{
			FileUtil.delete(this.originalSipFolder + LockFileSuffix);

			//	ToDo: How to set this.mode now?
//			this.mode = Mode.???;
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
			return false;
		}

		return true;
	}


	protected void lockAsSubmitRequestPending()
	{
		this.unlock();

		try
		{
			FileUtil.createFileWithContent(this.originalSipFolder + LockFileSuffix, SubmitStatus.getSubmitRequestPendingLockId());
			this.mode = Mode.Locked;
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
		}
	}

	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * Document mode:
	 *
	 * Undefined:	Temporary. The document is just being read; the mode will soon change.
	 * ReadWrite:	The user can change whatever he wants. This is the case when the SIP was opened with a working copy.
	 * NoFileOps:	The user can only change things that don't affect the file system. This is the case when the SIP was opened without a working copy.
	 * ReadOnly:	The user can not make any changes to the SIP. This is the case when the SIP was opened in ReadOnly mode.
	 * Locked:		The user can not make any changes to the SIP. This is the case when the SIP was opened by somebody else in ReadWrite or NoFileOps mode.
	 */
	public enum Mode
	{
		Undefined, ReadWrite, ReadWriteNoFileOps, ReadOnly, Locked
	}


	/**
	 * This class is needed within the class <a href="../Document.html">Document</a> to make the SAXReader generate an instance of the Document class.
	 *
	 * @author denis
	 */
	static private class DocumentFactory extends org.dom4j.DocumentFactory
	{
		// Factory methods
		@Override
		public Document createDocument()
		{
			ch.docuteam.docudarc.mets.Document document = new ch.docuteam.docudarc.mets.Document();
			document.setDocumentFactory(this);

			return document;
		}
	}

}
