/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.sa;

import java.io.*;
import java.net.*;
import java.util.*;

import org.dom4j.*;
import org.dom4j.io.SAXReader;
import org.dom4j.tree.DefaultDocument;

import ch.docuteam.docudarc.sa.archive.Archive;
import ch.docuteam.docudarc.sa.dss.DataSubmissionSession;
import ch.docuteam.docudarc.sa.producer.Producer;
import ch.docuteam.docutools.exception.ExceptionCollector;
import ch.docuteam.docutools.exception.ExceptionCollectorException;
import ch.docuteam.docutools.file.FileUtil;
import ch.docuteam.docutools.file.exception.FileIsNotADirectoryException;
import ch.docuteam.docutools.file.exception.FileUtilExceptionListException;
import ch.docuteam.docutools.os.SystemProcess;
import ch.docuteam.docutools.os.SystemProcessException;
import ch.docuteam.docutools.out.Logger;


/**
 * An instance of this class is the java-equivalent of a Submission Agreement file.
 * It contains
 * <ul>
 * <li>List&lt;<a href="./producer/Producer.html">Producer</a>&gt;</li>
 * <li><a href="./archive/Archive.html">Archive</a></li>
 * <li>Map&lt;String, <a href="./dss/DataSubmissionSession.html">DataSubmissionSession</a>&gt;</li>
 * </ul>
 * In Addition in contains an ID, a title, and a status.
 *
 * @author denis
 *
 */
public class SubmissionAgreement extends DefaultDocument
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String						URLForHTMLReplacementString		= "{saId}";
	static private final String						URLForHTMLExtension				= "/sites/" + URLForHTMLReplacementString + "/index.html";

	static private final String						URLForUpdateExtension			= "/contribXML/";

	static private final String						SAFolderDefault					= "agreements";
	static private final String						SAFolderTemp					= FileUtil.getTempFolder() + "/agreements";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	This is needed for "openAsHTMLExternally()" and "updateSAsFromServer()":
	//	Should be something like: "http://192.168.0.22/webjaxe";
	static private String							BaseURL;

	//	This folder contains the SAs:
	static private String							SAFolder = SAFolderDefault;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String									fileName;
	private String									saId;

	private String									title;
	private SubmissionAgreementStatus				status;

	private List<Producer>							producers;
	private Archive									archive;
	private Map<String, DataSubmissionSession>		dataSubmissionSessions;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	static public SubmissionAgreement read(String saFilePath) throws FileNotFoundException, DocumentException, ExceptionCollectorException
	{
		saFilePath = SAFolder + "/" + saFilePath;

		if (!new File(saFilePath).exists())		throw new FileNotFoundException(saFilePath);

		File saFile = new File(saFilePath);

		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new SubmissionAgreementFactory());
		SubmissionAgreement sa = (SubmissionAgreement)reader.read(saFile);

		ExceptionCollector.clear();
		sa.parse(saFilePath);
		if (!ExceptionCollector.isEmpty())		throw new ExceptionCollectorException();

		return sa;
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	/**
	 * 	This is needed for "openAsHTMLExternally()" and "updateSAsFromServer()" and should be something like: "http://192.168.0.22/webjaxe":
	 */
	static public void initializeBaseURL(String baseURL)
	{
		BaseURL = baseURL;
	}

	static public String getBaseURL()
	{
		return BaseURL;
	}


	/**
	 * Set the SA folder. The default is "./agreements".
	 */
	static public void setSAFolder(String newSAFolder)
	{
		SAFolder = newSAFolder;
	}

	static public String getSAFolder()
	{
		return SAFolder;
	}


	/**
	 * Overwrite the SAs with the version from the server.
	 * Return a list of SA file names that are on the client but NOT on the server, the caller might want to delete them.
	 * @throws IOException
	 * @throws FileUtilExceptionListException
	 * @throws java.lang.Exception
	 */
	static public List<String> updateSAsFromServer() throws IOException, FileUtilExceptionListException
	{
		if (BaseURL == null || BaseURL.isEmpty())		throw new MalformedURLException("BaseURL for retrieving SAs from the server is not initialized");

		URL saDirURL = new URL(BaseURL + URLForUpdateExtension);

		Logger.info("Updating SAs from URL: " + saDirURL);

		//	Get SA directory:
		HttpURLConnection connection = null;
		InputStream inputStream = null;
		StringBuilder content = new StringBuilder(5000);
		try
		{
			connection = (HttpURLConnection)saDirURL.openConnection();
			connection.connect();
			inputStream = (InputStream)connection.getContent();

			for (int c; (c = inputStream.read()) != -1;)		content.append((char)c);
		}
		finally
		{
			try
			{
				if (inputStream != null)	inputStream.close();
				if (connection != null)		connection.disconnect();
			}
			catch (IOException ex){}
		}

		//	Parse SA directory and collect SA names:
		Vector<String> saNames = new Vector<String>(20);
		int foundEnd = 0;
		do
		{
			int foundStart = content.indexOf("<li><a href=", foundEnd);
			if (foundStart == -1)			break;
			foundEnd = content.indexOf("/\"> ", foundStart);
			if (foundEnd == -1)				break;
			String saName = content.substring(foundStart + 13, foundEnd).trim();
			if (!saName.startsWith("/"))	saNames.add(saName);
		}
		while (foundEnd != -1);

		File saFolderTemp = new File(SAFolderTemp);
		try
		{
			//	Create temporary SA Folder:
			FileUtil.delete(saFolderTemp);
			saFolderTemp.mkdirs();

			//	Download SAs into temporary folder:
			for (String saName: saNames)
			{
				String saFileName = saName + ".xml";

				Logger.debug("Downloading SA: " + saFileName);

				URL saURL = null;
				try
				{
					saURL = new URL(saDirURL + saName + "/" + saFileName);
				}
				catch (MalformedURLException ex)
				{
					Logger.warn("Exception in URL <" + saURL + ">: " + ex);
					continue;
				}

				File file = new File(SAFolderTemp + "/" + saFileName);
				FileOutputStream fileOutputStream = null;
				try
				{
					fileOutputStream = new FileOutputStream(file);
					connection = (HttpURLConnection)saURL.openConnection();
					connection.connect();
					inputStream = (InputStream)connection.getContent();

					//	Copy byte by byte:
					for (int c; (c = inputStream.read()) != -1;)		fileOutputStream.write(c);
				}
				finally
				{
					try
					{
						if (fileOutputStream != null)		fileOutputStream.close();
						if (inputStream != null)			inputStream.close();
						if (connection != null)				connection.disconnect();
					}
					catch (IOException ex){}
				}
			}

			//	Nothing was downloaded:
			if (saFolderTemp.list().length == 0)		return new ArrayList<String>();

			//	Download is complete. Now compare existing SAs with the downloaded ones and collect those being in excess and hence are subject for deletion:
			List<String> saNamesToBeDeleted = new Vector<String>(10);
			File saFolder = new File(SAFolder);
			if (!saFolder.exists())		saFolder.mkdirs();
			for (String s: saFolder.list(new FilenameFilter() { @Override public boolean accept(File dir, String name) { return name.endsWith(".xml") || name.endsWith(".XML"); }}))
			{
				if (!new File(SAFolderTemp + "/" + s).exists())		saNamesToBeDeleted.add(s);
			}

			//	Now copy the downloaded SAs from the temporary folder to the SA folder:
			for (File f: saFolderTemp.listFiles())
			{
				Logger.debug("Copying/overwriting: " + f);
				FileUtil.copyToFolderOverwriting(f, saFolder);
			}

			return saNamesToBeDeleted;
		}
		finally
		{
			//	Finally delete the temporary folder:
			FileUtil.delete(saFolderTemp);
		}
	}


	/**
	 * Return a list of Overviews of all final SAs in my SAFolder.
	 * @return
	 */
	static public List<Overview>getAllFinalOverviews()
	{
		List<Overview>saOverviewList = new Vector<Overview>(20);

		for (String fileName: new File(SAFolder).list())
		{
			if (!(fileName.endsWith(".xml") || fileName.endsWith(".XML")))		continue;
			if (fileName.startsWith(".")    || fileName.startsWith("~"))		continue;

			Logger.debug("Retrieving overviews for SA file: '" + fileName + "'");

			try
			{
				SubmissionAgreement sa = SubmissionAgreement.read(fileName);
				if (sa.isFinal())
				{
					List<Overview> overviews = sa.getOverviews();
					Logger.debug("Found " + overviews.size() + " overviews");
					saOverviewList.addAll(overviews);
				}
				else
					Logger.debug("Skipping SA '" + fileName + "' because it is not final");
			}
			catch (ExceptionCollectorException ex)
			{
				Logger.warn("Ignoring SA '" + fileName + "' because of parsing errors: " + ex.getMessage());
				continue;
			}
			catch (java.lang.Exception ex)
			{
				ex.printStackTrace();
				continue;
			}
		}

		return saOverviewList;
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getFileName()
	{
		return this.fileName;
	}

	public String getSaId()
	{
		return this.saId;
	}

	public String getTitle()
	{
		return this.title;
	}

	public SubmissionAgreementStatus getStatus()
	{
		return this.status;
	}

	public List<Producer> getProducers()
	{
		return this.producers;
	}

	public Archive getArchive()
	{
		return this.archive;
	}

	public Map<String, DataSubmissionSession> getDataSubmissionSessions()
	{
		return this.dataSubmissionSessions;
	}

	public DataSubmissionSession getDataSubmissionSession(String dssID)
	{
		if (!this.dataSubmissionSessions.containsKey(dssID))		throw new IndexOutOfBoundsException("The Submission Agreement '" + this.saId + "' doesn't contain the Data Submission Session '" + dssID + "'");

		return this.dataSubmissionSessions.get(dssID);
	}

	public List<Overview> getOverviews()
	{
		return Overview.createOverviews(this);
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean allowsFile(String dssId, String filePath) throws FileNotFoundException
	{
		return this.getDataSubmissionSession(dssId).allowsFile(filePath);
	}

	public boolean allowsFileType(String dssId, String filePath, String puid, String mimeType)
	{
		return this.getDataSubmissionSession(dssId).allowsFileType(filePath, puid, mimeType);
	}

	public boolean isFinal()
	{
		return SubmissionAgreementStatus.get("Final") == this.getStatus();
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	/**
	 * Open the SA with the name passed in an external browser, using the passed url as the location.
	 * @param urlWithParameter
	 * @param saId
	 * @throws IOException
	 * @throws MalformedURLException
	 * @throws SystemProcessException
	 * @throws InterruptedException
	 * @throws FileIsNotADirectoryException
	 * @throws URISyntaxException
	 */
	public void openAsHTMLExternally() throws MalformedURLException, IOException, InterruptedException, SystemProcessException, FileIsNotADirectoryException, URISyntaxException
	{
		if (BaseURL == null)		throw new MalformedURLException("BaseURL for retrieving SAs from the server is not initialized");

		//	Replace argument "{saId}" in urlWithParameter by the actual parameter saId:
		String actualURL = (BaseURL + URLForHTMLExtension).replace(URLForHTMLReplacementString, this.saId);

		Logger.debug("Opening Submission Agreement URL externally: " + actualURL);

		//	Check if this URL exists, throw exception if not:
		new URL(actualURL).getContent();

		SystemProcess.openExternally(actualURL);
	}

	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------

	private SubmissionAgreement parse(String saFilePath)
	{
		this.fileName = saFilePath;

		Element root = this.getRootElement();
		this.saId = root.attributeValue("saID");
		this.title = root.attributeValue("title");
		this.status = SubmissionAgreementStatus.get(root.attributeValue("status"));

		this.producers = Producer.parse(this, root);
		this.archive = Archive.parse(this, root);
		this.dataSubmissionSessions = DataSubmissionSession.parse(this, root);

		return this;
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder stringBuilder = new StringBuilder("\n====\tSubmissionAgreement on '")
			.append(this.fileName)
			.append("'\t====\n\t")
			.append(this.saId)
			.append("/")
			.append(this.title)
			.append("/")
			.append(this.status);

		for(Producer p: this.producers)		stringBuilder.append("\n\t\t").append(p);

		stringBuilder.append("\n\t\t").append(this.archive);

		for(DataSubmissionSession d: this.dataSubmissionSessions.values())		stringBuilder.append("\n\t\t").append(d);

		return stringBuilder.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	static public class Overview implements Comparable<Overview>
	{
		public String						saId;
		public String						title;
		public String						fileName;
		public SubmissionAgreementStatus	status;
		public String						dssId;
		public String						dssTitle;


		Overview(SubmissionAgreement sa, DataSubmissionSession dss)
		{
			this.saId = sa.saId;
			this.title = sa.title;
			this.fileName = sa.fileName;
			this.status = sa.status;
			this.dssId = dss.getDssID();
			this.dssTitle = dss.getTitle();
		}


		static List<Overview> createOverviews(SubmissionAgreement sa)
		{
			List<Overview> overviews = new ArrayList<Overview>(sa.dataSubmissionSessions.size());
			for (DataSubmissionSession dss: sa.dataSubmissionSessions.values())		overviews.add(new Overview(sa, dss));

			Collections.sort(overviews);
			return overviews;
		}


		public SubmissionAgreement getSubmissionAgreement() throws FileNotFoundException, DocumentException, ExceptionCollectorException
		{
			return SubmissionAgreement.read(this.saId + ".xml");
		}


		@Override
		public String toString()
		{
			return this.title + "/" + this.dssTitle + "\t    (" + this.saId + "/" + this.dssId + ")";
		}


		private String compareString()
		{
			return this.title + this.dssTitle;
		}


		/* (non-Javadoc)
		 * @see java.lang.Comparable#compareTo(java.lang.Object)
		 */
		@Override
		public int compareTo(Overview o)
		{
			return this.compareString().compareTo(o.compareString());
		}
	}


	/**
	 * This class is needed within the class <a href="./SubmissionAgreement.html">SubmissionAgreement</a> to make the SAXReader generate an instance of the SubmissionAgreement class.
	 *
	 * @author denis
	 */
	static private class SubmissionAgreementFactory extends org.dom4j.DocumentFactory
	{
		// Factory methods
		@Override
		public Document createDocument()
		{
			Map<String, String> namespaces = new TreeMap<String, String>();
			namespaces.put("SA", "http://www.docuteam.ch/xmlns/submAgr");
			this.setXPathNamespaceURIs(namespaces);

			SubmissionAgreement sa = new SubmissionAgreement();
			sa.setDocumentFactory(this);

			return sa;
		}
	}

}
