/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.sa.dss;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import org.dom4j.Element;

import uk.gov.nationalarchives.droid.core.interfaces.IdentificationResult;
import ch.docuteam.docudarc.sa.SubmissionAgreement;
import ch.docuteam.docudarc.sa.common.NodeAbstract;
import ch.docuteam.docutools.file.MetadataProviderDROID;
import ch.docuteam.docutools.out.Logger;


/**
 * This class, used by <a href="./DataSubmissionSession.html">DataSubmissionSession</a>, represents a list of file formats this data submission session allows.
 * First the PUID is checked, then the mimeType, and finally the file extension.
 * If none of these criteria matches, the FileFormats can be strict.
 * In this case, false is returned, meaning that this file type is not allowed.
 * If the FileFormat is not strict, true is returned even though this file type is not allowed.
 *
 * @author denis
 *
 */
public class FileFormats extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private boolean				isStrict = true;
	private List<String>		names = new ArrayList<String>();
	private List<String>		puids = new ArrayList<String>();
	private List<String>		mimeTypes = new ArrayList<String>();
	private List<String>		extensions = new ArrayList<String>();

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	private FileFormats(SubmissionAgreement sa, Element element)
	{
		this.sa = sa;
		this.element = element;

		String strict = element.attributeValue("strict");
		this.isStrict = (strict == null || strict.isEmpty()? true: new Boolean(strict));

		for (Object o: element.selectNodes("./SA:name"))			this.names.add(((Element)o).getText());
		for (Object o: element.selectNodes("./SA:puid"))			this.puids.add(((Element)o).getText().toLowerCase());
		for (Object o: element.selectNodes("./SA:mimetype"))		this.mimeTypes.add(((Element)o).getText().toLowerCase());
		for (Object o: element.selectNodes("./SA:extension"))		this.extensions.add(((Element)o).getText().toLowerCase());
	}

	//	========	Static Public			=======================================================

	/**
	 * Package visibility because this method is called by DataSubmissionSession.
	 * @param sa
	 * @param parent
	 * @return
	 */
	static FileFormats parse(SubmissionAgreement sa, Element parent)
	{
		Element fileFormatsElement = (Element)parent.selectSingleNode("./SA:fileFormats");
		if (fileFormatsElement == null)		return null;

		return new FileFormats(sa, fileFormatsElement);
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public boolean isStrict()
	{
		return this.isStrict;
	}

	public List<String> getNames()
	{
		return this.names;
	}

	public List<String> getPuids()
	{
		return this.puids;
	}

	public List<String> getMimeTypes()
	{
		return this.mimeTypes;
	}

	public List<String> getExtensions()
	{
		return this.extensions;
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean allowsFile(String dssId, String filePath) throws FileNotFoundException
	{
		if (!new File(filePath).exists())		throw new FileNotFoundException(filePath);

		String puid = null;
		String mimeType = null;

		IdentificationResult identificationResult = null;
		try
		{
			identificationResult = MetadataProviderDROID.getIdentificationResult(filePath);
		}
		catch (Exception ex){}		//	Ignore exceptions

		//	It can actually happen that fileFormatHit is null:
		if (identificationResult != null)
		{
			puid = identificationResult.getPuid();
			mimeType = identificationResult.getMimeType();
		}

		return this.allowsFileType(dssId, filePath, puid, mimeType);
	}


	public boolean allowsFileType(String dssID, String filePath, String puid, String mimeType)
	{
		String extension = filePath.substring(filePath.lastIndexOf(".") + 1);

		if ((puid != null		&& !puid.isEmpty()		&& this.puids.contains(puid.toLowerCase()))
		||	(mimeType != null	&& !mimeType.isEmpty()	&& this.mimeTypes.contains(mimeType.toLowerCase()))
		||	(extension != null	&& !extension.isEmpty()	&& this.extensions.contains(extension.toLowerCase())))			return true;

		if (this.isStrict)
		{
			Logger.debug("Submission Agreement '" + this.sa.getSaId() + "' Data Submission Session '" + dssID + "' doesn't allow file '" + filePath + "(" + puid + ":" + mimeType + ")'");
			return false;
		}

		Logger.debug("Submission Agreement '" + this.sa.getSaId() + "' Data Submission Session '" + dssID + "' doesn't allow file '" + filePath + "(" + puid + ":" + mimeType + ")', but since it is not strict, let it pass.");
		return true;
	}

	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return new StringBuilder("[FileFormat:")
			.append(this.isStrict + "/")
			.append(this.names + "/")
			.append(this.puids + "/")
			.append(this.mimeTypes + "/")
			.append(this.extensions + "]")
			.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
