/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.util;

import java.io.*;
import java.net.*;
import java.nio.charset.Charset;

import ch.docuteam.docudarc.exceptions.VirusScannerClamException;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.out.Tracer;

/**
 * Utility class to send a limited set of commands to a ClamAV daemon.
 *
 * The class has to be initialized once with at least a host and port parameter,
 * after which the respective command methods can be used.
 *
 * @author Denis Lemberger, Andreas Nef
 *
 */
public abstract class VirusScannerClam
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final int			DefaultSocketTimeout = 60000;
	static private final int			DefaultChunkSize = 4096;

	/**
	 * The commands are according to the clamav webiste starting with the letter
	 * 'z' and therefore have to end with the null character.
	 *
	 * @see <a href=" http://www.clamav.net/doc/latest/html/node28.html">clamav
	 *      documentation</a>
	 */
	static private final byte[]			CmdInstream = "zINSTREAM\0".getBytes();
	static private final byte[]			CmdScan = "zSCAN ${file}\0".getBytes();
	static private final String			ResponseOk = "OK";
	static private final String			ResponseVirusFound = "FOUND";
	static private final String			ResponseError = "lstat() failed: No such file or directory. ERROR";

	static private final byte[]			CmdPing = "zPING\0".getBytes();
	static private final String			ResponsePong = "PONG";

	static private final byte[]			CmdVersion = "zVERSION\0".getBytes();
	static private final String			ResponseVersion = "ClamAV";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private String				ClamAVHost = null;
	static private int					ClamAVPort = 0;
	static private int					SocketTimeout = DefaultSocketTimeout;
	static private Charset				SocketCharset = Charset.defaultCharset();

	static private String				Response = "";

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * @param args
	 */
	static public void main(String[] args) throws Exception
	{
		initialize("localhost", 3310);
		Tracer.trace(ping());
		Tracer.trace(version());
		Tracer.trace(scan("src/ch/docuteam/unittest/files/VirusScanner/NoVirus.txt"));
		Tracer.trace(scan("src/ch/docuteam/unittest/files/VirusScanner/InfectedWithTestVirus.txt"));
		Tracer.trace(scan("src/ch/docuteam/unittest/files/VirusScanner/missing.txt"));
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	static public void initialize(String host, int port)
	{
		ClamAVHost = host;
		ClamAVPort = port;
	}


	static public void initialize(String host, int port, int socketTimeoutInSecs)
	{
		initialize(host, port);
		SocketTimeout = 1000 * socketTimeoutInSecs;
	}

	static public void setCharset(String charsetName) {
		SocketCharset = Charset.forName(charsetName);
	}

	//	--------		Business Ops		-------------------------------------------------------

	static public boolean ping() throws IOException, SocketException, VirusScannerClamException {
		Logger.info("Sending ping to ClamAV...");

		sendCommand(CmdPing);

		if (Response.equals(ResponsePong))			return true;

		if (Response.isEmpty())						throw new VirusScannerClamException("Empty response");

		//	In any other case, an error occurred:
		throw new VirusScannerClamException(Response);
	}

	static public boolean version() throws IOException, SocketException, VirusScannerClamException {
		Logger.info("Getting ClamAV version...");

		sendCommand(CmdVersion);

		if (Response.startsWith(ResponseVersion))	return true;

		if (Response.isEmpty())						throw new VirusScannerClamException("Empty response");

		//	In any other case, an error occurred:
		throw new VirusScannerClamException(Response);
	}

	/**
	 * This scan will send the absolute path of the file to the clamav daemon
	 * using the SCAN command. The user which the daemon is running under has to
	 * have access to this file.
	 *
	 * @param fileName
	 * @return
	 * @throws SocketException
	 * @throws IOException
	 * @throws VirusScannerClamException
	 */
	static public boolean scan(String fileName) throws SocketException, IOException, VirusScannerClamException
	{
		return scan(new File(fileName));
	}

	/**
	 * This scan will send the absolute path of the file to the clamav daemon
	 * using the SCAN command. The user which the daemon is running under has to
	 * have access to this file.
	 *
	 * @param fileName
	 * @return
	 * @throws SocketException
	 * @throws IOException
	 * @throws VirusScannerClamException
	 */
	static public boolean scan(File file) throws SocketException, IOException, VirusScannerClamException
	{
		Logger.info("Scanning file: " + file.getPath());

		Logger.debug("Using charset: " + SocketCharset.displayName());
		sendCommand(new String(CmdScan).replace("${file}", file.getAbsolutePath()).getBytes(SocketCharset));

		if (Response.isEmpty())						throw new VirusScannerClamException("Empty response");

		if (Response.endsWith(ResponseError))		throw new VirusScannerClamException("Could not access file");

		if (Response.endsWith(ResponseOk))			return true;
		if (Response.endsWith(ResponseVirusFound))	return false;

		//	In any other case, an error occurred:
		throw new VirusScannerClamException(Response);
	}

	/**
	 * This scan will stream the file to the clamav daemon using the STREAM command.
	 *
	 * @param fileName
	 * @return
	 * @throws SocketException
	 * @throws IOException
	 * @throws VirusScannerClamException
	 */
	static public boolean scanStreaming(File file) throws SocketException, IOException, VirusScannerClamException
	{
		return scanStreaming(new FileInputStream(file));
	}

	//	--------		Accessing			-------------------------------------------------------

	static public String getResponse()
	{
		return Response;
	}

	//	========	Static Private			=======================================================

	static private void sendCommand(byte[] command) throws IOException, VirusScannerClamException {
		if (ClamAVHost == null || ClamAVPort == 0)		throw new VirusScannerClamException("VirusScannerClam is not initialized");
		Response = "";

		Socket socket = new Socket();
		try
		{
			socket.connect(new InetSocketAddress(ClamAVHost, ClamAVPort));
			socket.setSoTimeout(SocketTimeout);
			socket.setKeepAlive(true);
		}
		catch (ConnectException x)
		{
			if (socket != null)		socket.close();

			Logger.error("A Socket Connect Exception occurred - maybe the clamd virus check service is not running?");
			throw x;
		}

		int bytesRead = DefaultChunkSize;
		byte[] byteBuffer = new byte[DefaultChunkSize];
		DataOutputStream dataOutputStream = null;

		try
		{
			dataOutputStream = new DataOutputStream(socket.getOutputStream());
			dataOutputStream.write(command);

			//	Read the socket input stream:
			bytesRead = socket.getInputStream().read(byteBuffer);
			if (bytesRead > 0)		Response = new String(byteBuffer, 0, bytesRead).trim();
		}
		finally
		{
			if (dataOutputStream != null)			dataOutputStream.close();
			if (socket != null)						socket.close();
		}

		Logger.info("Response: " + Response);
	}

	static private boolean scanStreaming(InputStream inputStream) throws IOException, SocketException, VirusScannerClamException
	{
		if (ClamAVHost == null || ClamAVPort == 0)		throw new VirusScannerClamException("VirusScannerClam is not initialized");

		Socket socket = new Socket();
		try
		{
			socket.connect(new InetSocketAddress(ClamAVHost, ClamAVPort));
			socket.setSoTimeout(SocketTimeout);
			socket.setKeepAlive(true);
		}
		catch (ConnectException x)
		{
			if (socket != null)		socket.close();

			Logger.error("A Socket Connect Exception occurred - maybe the clamd virus check service is not running?");
			throw x;
		}

		DataOutputStream dataOutputStream = null;
		Response = "";

		int bytesRead = DefaultChunkSize;
		byte[] byteBuffer = new byte[DefaultChunkSize];

		//	Copy the input stream to the socket output stream:
		try
		{
			dataOutputStream = new DataOutputStream(socket.getOutputStream());
			dataOutputStream.write(CmdInstream);

			while (bytesRead == DefaultChunkSize)
			{
				bytesRead = inputStream.read(byteBuffer);
				Logger.debug("Bytes read: " + bytesRead);

				if (bytesRead == -1)		break;

				// we may exceed the clamd size limit, so we don't immediately return if we get an error here.
				dataOutputStream.writeInt(bytesRead);
				dataOutputStream.write(byteBuffer, 0, bytesRead);
			}

			dataOutputStream.writeInt(0);
			dataOutputStream.flush();

			//	Read the socket input stream:
			bytesRead = socket.getInputStream().read(byteBuffer);
			if (bytesRead > 0)		Response = new String(byteBuffer, 0, bytesRead).trim();
		}
		finally
		{
			if (dataOutputStream != null)			dataOutputStream.close();
			if (socket != null)						socket.close();
		}

		Logger.info("Response: " + Response);

		if (Response.isEmpty())						throw new VirusScannerClamException("Empty response");

		if (Response.endsWith(ResponseOk))			return true;
		if (Response.endsWith(ResponseVirusFound))	return false;

		//	In any other case, an error occurred:
		throw new VirusScannerClamException(Response);
	}

}
