/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docupack.gui;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.util.*;
import java.util.List;

import javax.media.jai.JAI;
import javax.swing.*;

import org.icepdf.ri.common.SwingController;
import org.icepdf.ri.common.SwingViewBuilder;

import ch.docuteam.docudarc.mets.structmap.NodeAbstract;
import ch.docuteam.docudarc.util.FileConverter;
import ch.docuteam.docutools.file.FileUtil;
import ch.docuteam.docutools.file.exception.FileUtilExceptionListException;
import ch.docuteam.docutools.id.UniqueID;
import ch.docuteam.docutools.os.SystemProcess;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.string.StringUtil;
import ch.docuteam.docutools.translations.I18N;

import com.sun.jimi.core.Jimi;

/**
 * @author denis
 *
 */
public class FilePreviewer extends JPanel
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private JFrame							filePreviewFrame;
	private JPanel							filePreviewPanelContainer;

	private JButton							openFileExternallyButton;
	private JButton							openFileSeparatelyButton;

	private NodeAbstract					node;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	protected FilePreviewer()
	{
		super(new BorderLayout());

		this.openFileExternallyButton = new JButton(I18N.translate("ButtonOpenFileExternally"), new ImageIcon("resources/images/View.png"));
		this.openFileExternallyButton.setEnabled(false);
		this.openFileExternallyButton.setToolTipText(I18N.translate("ToolTipOpenFileExternally"));
		this.openFileExternallyButton.addActionListener(
				new ActionListener() { @Override public void actionPerformed(ActionEvent e) { FilePreviewer.this.openFileExternallyButtonClicked(FilePreviewer.this.node); }});

		this.openFileSeparatelyButton = new JButton(I18N.translate("ButtonOpenFileSeparately"), new ImageIcon("resources/images/PreviewInWindow.png"));
		this.openFileSeparatelyButton.setEnabled(false);
		this.openFileSeparatelyButton.setToolTipText(I18N.translate("ToolTipOpenFileSeparately"));
		this.openFileSeparatelyButton.addActionListener(
				new ActionListener() { @Override public void actionPerformed(ActionEvent e) { FilePreviewer.this.openFileSeparatelyButtonClicked(FilePreviewer.this.node); }});

		this.filePreviewPanelContainer = new JPanel(new BorderLayout());
		this.filePreviewPanelContainer.add(new JPanel());

		Box buttonBox = new Box(BoxLayout.X_AXIS);
		buttonBox.add(Box.createHorizontalGlue());
		buttonBox.add(this.openFileExternallyButton);
		buttonBox.add(this.openFileSeparatelyButton);
		buttonBox.add(Box.createHorizontalGlue());

		this.add(buttonBox, BorderLayout.NORTH);
		this.add(this.filePreviewPanelContainer, BorderLayout.CENTER);

		//	Cleanup the temp-folder now - under Windows, he fails cleaning-up on shutdown:
		FilePreviewPanel.cleanupTemporaryFolder();
	}

	//	========	Static Public			=======================================================

	static public void setCacheSizeLimit(int cacheSizeLimit)
	{
		FilePreviewPanel.setCacheSizeLimit(cacheSizeLimit);
	}

	static public int getCacheSizeLimit()
	{
		return FilePreviewPanel.getCacheSizeLimit();
	}

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public void setNode(NodeAbstract node)
	{
		this.node = node;

		this.filePreviewPanelContainer.removeAll();
		if (this.node != null)		this.filePreviewPanelContainer.add(FilePreviewPanel.create(node, this), BorderLayout.CENTER);

		this.enableOrDisableButtons();
	}

	//	--------		Inquiring			-------------------------------------------------------

	public boolean isPreviewInSeparateWindow()
	{
		return this.filePreviewFrame != null && this.filePreviewFrame.isVisible();
	}

	//	--------		Interface			-------------------------------------------------------

	/**
	 * This method is public so that it can be called from outside (e.g. by a double-click on a node).
	 */
	public void openFileExternallyButtonClicked(NodeAbstract node)
	{
		if (node == null)			return;
		if (node.isFolder())		return;

		String tempFileName = FileUtil.getTempFolder() + "/" + node.getFile().getName();
		try
		{
			if (!new File(tempFileName).exists())		FileUtil.copyToOverwriting(node.getAbsolutePathString(), tempFileName);

			SystemProcess.openExternally(tempFileName);
		}
		catch (Exception e)
		{
			JOptionPane.showMessageDialog(this, e.toString(), I18N.translate("TitleCantOpenFileExternally"), JOptionPane.ERROR_MESSAGE);
		}
		finally
		{
			new File(tempFileName).deleteOnExit();
		}
	}


	@Override
	public void repaint()
	{
		if (this.isPreviewInSeparateWindow())
		{
			//	If the preview is in the separate window, repaint it:
			this.filePreviewFrame.validate();
			this.filePreviewFrame.repaint();
		}
		else
		{
			//	Otherwise repaint me:
			super.repaint();
		}
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------

	private void openFileSeparatelyButtonClicked(NodeAbstract node)
	{
		//	Open a new JFrame containing the filePreviewPanelContainer:

		if (this.filePreviewFrame == null)
		{
			this.filePreviewFrame = new JFrame();
			this.filePreviewFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
			this.filePreviewFrame.addWindowListener(
					new WindowAdapter() { @Override public void windowClosing(WindowEvent e) { FilePreviewer.this.closeWindowButtonClicked(); }});

			this.filePreviewFrame.setSize(new Dimension(800, 800));
			this.filePreviewFrame.setLocationRelativeTo(null);
		}

		//	If it is already open, bring it to the front:
		if (this.filePreviewFrame.isVisible())
		{
			this.filePreviewFrame.toFront();
			this.filePreviewFrame.requestFocus();
			return;
		}

		this.remove(this.filePreviewPanelContainer);
		this.validate();
		this.repaint();

		this.filePreviewFrame.add(this.filePreviewPanelContainer);
		this.filePreviewFrame.setVisible(true);
		this.filePreviewFrame.toFront();
		this.filePreviewFrame.requestFocus();
	}


	private void closeWindowButtonClicked()
	{
		//	Close the separate JFrame and place the filePreviewPanelContainer back into myself:

		this.filePreviewFrame.setVisible(false);
		this.filePreviewFrame.remove(this.filePreviewPanelContainer);

		this.add(this.filePreviewPanelContainer, BorderLayout.CENTER);
		this.validate();
		this.repaint();

		this.enableOrDisableButtons();
	}

	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	private void enableOrDisableButtons()
	{
		//	If the current node is null, disable all buttons:
		if (this.node == null)
		{
			this.openFileExternallyButton.setEnabled(false);
			this.openFileSeparatelyButton.setEnabled(false);
			return;
		}

		//	If the current node is a folder, disable the "openFileExternallyButton":
		if (this.node.isFolder())
			this.openFileExternallyButton.setEnabled(false);
		else
			this.openFileExternallyButton.setEnabled(true);

		this.openFileSeparatelyButton.setEnabled(true);
	}

	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------


	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	//	Package visibility, must be seen by MainView:
	static class FilePreviewPanel extends JPanel
	{
		//	========	Static Final Private	=======================================================

		static private final String				TempPDFFileNameTemplate = FileUtil.getTempFolder() + "/FileConverter/Temp_%s.pdf";
		static private final String				TempPNGFileNameTemplate = FileUtil.getTempFolder() + "/FileConverter/Temp_%s.png";

		static private final int				DefaultCacheSizeLimit = 100;

		static private final String				Tab = "&nbsp;&nbsp;&nbsp;&nbsp;";

		//	========	Static Private			=======================================================

		//	The temporary files:
		static private List<String>				TempPDFFileNames = new Vector<String>();
		static private List<String>				TempPNGFileNames = new Vector<String>();

		//	This is the cache:
		static private Map<NodeAbstract, FilePreviewPanel>	Cache = new HashMap<NodeAbstract, FilePreviewPanel>();

		//	This is the default cacheSize limit - the actual limit can be changed using setCacheSizeLimit():
		static private int						CacheSizeLimit = DefaultCacheSizeLimit;

		//	This list contains the order in which the cache elements have been retrieved:
		static private List<NodeAbstract>		CacheHitOrder = new ArrayList<NodeAbstract>(CacheSizeLimit);

		//	========	Constructors Private	=======================================================

		private FilePreviewPanel(NodeAbstract node)
		{
			super(new BorderLayout());

			try
			{
				this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

				if (node == null)
				{
					Logger.debug("No item selected");

					this.add(new JPanel());
					return;
				}

				if (!node.fileExists() || !node.canRead())
				{
					Logger.debug("Item '" + node.getLabel() + "' could not be read");

					JLabel previewNotAvailable = new JLabel(I18N.translate("LabelPreviewCantReadFile"), SwingConstants.CENTER);
					previewNotAvailable.setOpaque(true);
					previewNotAvailable.setBackground(Color.LIGHT_GRAY);
					this.add(previewNotAvailable);

					return;
				}

				String fileName = node.getFile().getAbsolutePath();
				FileType fileType = FileType.check(node);

				switch (fileType)
				{
					case Folder:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as folder");

						StringBuilder pathString = new StringBuilder("<html>" + I18N.translate("LabelPreviewFolder") + "<br>");
						int indent = 0;
						for (String pathElement: StringUtil.split(node.getPathString(), "/"))
						{
							pathString.append("<br>");
							for (int i = 0; i < indent; i++)		pathString.append(Tab);
							pathString.append(pathElement);
							indent++;
						}
						pathString.append("</html>");

						JLabel previewFolder = new JLabel(pathString.toString());
						previewFolder.setHorizontalAlignment(SwingConstants.CENTER);
						previewFolder.setOpaque(true);
						this.add(new JScrollPane(previewFolder));
						break;
					}
					case Text:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as text file");

						JTextArea previewText = new JTextArea();
						previewText.setEditable(false);
						previewText.setLineWrap(true);
						previewText.setWrapStyleWord(true);
						previewText.setTabSize(4);
						previewText.setText(FileUtil.getFileContentAsString(fileName));
						previewText.setCaretPosition(0);
						this.add(new JScrollPane(previewText));
						break;
					}
					case HTML:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as HTML file");

						JLabel previewHTML = new JLabel();
						previewHTML.setVerticalAlignment(SwingConstants.TOP);
						previewHTML.setText(FileUtil.getFileContentAsString(fileName));
						this.add(new JScrollPane(previewHTML));
						break;
					}
					case PDF:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as PDF file");

						this.add(new PDFPreviewer(this, fileName));
						break;
					}
					case OOConvertable:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as OO file");

						if (FileConverter.OOConverter.shallNotBeUsed())
						{
							JLabel previewNotAvailable = new JLabel(I18N.translate("LabelOOConverterNotAvailable"), SwingConstants.CENTER);
							previewNotAvailable.setOpaque(true);
							previewNotAvailable.setBackground(Color.LIGHT_GRAY);
							this.add(previewNotAvailable);
							break;
						}

						try
						{
							String tempPDFFileName = String.format(TempPDFFileNameTemplate, UniqueID.getString());
							FileConverter.OOConverter.convert2PDF(fileName, tempPDFFileName);

							TempPDFFileNames.add(tempPDFFileName);
							this.add(new PDFPreviewer(this, tempPDFFileName));
						}
						catch (Exception e)
						{
							e.printStackTrace();
							this.add(new JScrollPane(createPreviewError(e.toString())));
						}
						break;
					}
					case GraphicsNative:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as native graphics file");

						this.add(new GraphicsPreviewer(this, new ImageIcon(fileName)));
						break;
					}
					case GraphicsJAIConvertable:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as JAI graphics file");

						try
						{
							this.add(new GraphicsPreviewer(this, new ImageIcon(JAI.create("fileload", fileName).getAsBufferedImage())));
						}
						catch (Exception e)
						{
							e.printStackTrace();
							this.add(new JScrollPane(createPreviewError(e.toString())));
						}
						break;
					}
					case GraphicsJIMIConvertable:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as JIMI graphics file");

						try
						{
							this.add(new GraphicsPreviewer(this, new ImageIcon(Jimi.getImage(fileName))));
						}
						catch (Exception e)
						{
							e.printStackTrace();
							this.add(new JScrollPane(createPreviewError(e.toString())));
						}
						break;
					}
					case GraphicsImageMagickConvertable:
					{
						Logger.debug("Item '" + node.getLabel() + "' identified as ImageMagick graphics file");

						try
						{
							String tempPNGFileName = String.format(TempPNGFileNameTemplate, UniqueID.getString());

							// TODO: Do not use absolute path to executable, use migration-config.xml
							String[] commandLine = new String[] {"/opt/local/bin/convert", "-compress", "none", fileName, new File(tempPNGFileName).getAbsolutePath()};
							int errorCode = SystemProcess.execute(commandLine);
							if (errorCode != 0)			throw new Exception("File conversion of '" + fileName + "' to PNG format returned error code: " + errorCode);

							TempPNGFileNames.add(tempPNGFileName);
							this.add(new GraphicsPreviewer(this, new ImageIcon(tempPNGFileName)));
						}
						catch (Exception e)
						{
							e.printStackTrace();
							this.add(new JScrollPane(createPreviewError(e.toString())));
						}
						break;
					}
					case Unknown:
					default:
					{
						Logger.debug("Item '" + node.getLabel() + "' could not identify type");

						JLabel previewNotAvailable = new JLabel(I18N.translate("LabelPreviewNotAvailable"), SwingConstants.CENTER);
						previewNotAvailable.setOpaque(true);
						previewNotAvailable.setBackground(Color.LIGHT_GRAY);
						this.add(previewNotAvailable);
						break;
					}
				}
			}
			finally
			{
				this.validate();
				this.repaint();

				this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			}
		}

		//	========	Static Public			=======================================================

		/**
		 * Create a new FilePreviewPanel for the node passed, or take it from the cache if it is already in there:
		 */
		static public FilePreviewPanel create(NodeAbstract node, JPanel container)
		{
			//	If cache is turned off:
			if (CacheSizeLimit == 0)		return new FilePreviewPanel(node);

			if (Cache.containsKey(node))
			{
				//	Cached:
				Logger.info("Cache hit: " + node.getLabel());

				//	Remove this node from the CacheHitOrder list, wherever it is (it will be inserted at the beginning of the list later):
				CacheHitOrder.remove(CacheHitOrder.indexOf(node));
			}
			else
			{
				//	Not yet cached:
				if (CacheHitOrder.size() >= CacheSizeLimit)
				{
					//	Cache limit size reached: remove "oldest" node:
					Logger.debug("Dropped from cache: " + CacheHitOrder.get(0).getLabel());

					Cache.remove(CacheHitOrder.get(0));
					CacheHitOrder.remove(0);
				}

				//	Add node to the cache:
				Logger.debug("Insert into cache: " + node.getLabel());

				Cache.put(node, new FilePreviewPanel(node));
			}

			//	Put this node at the beginning of the CacheHitOrder list:
			CacheHitOrder.add(node);

			Logger.debug("Cache size: " + Cache.size() + "/" + CacheSizeLimit);

			return Cache.get(node);
		}


		static public void setCacheSizeLimit(int newCacheSizeLimit)
		{
			if (newCacheSizeLimit < 0)		throw new IllegalArgumentException("Cache size limit can not be set to a negative value");

			Logger.debug("Setting FilePreviewer cache size to " + newCacheSizeLimit);

			if (newCacheSizeLimit == 0)
			{
				Cache.clear();
				CacheHitOrder.clear();
			}
			else if (newCacheSizeLimit < CacheSizeLimit)
			{
				//	Remove "oldest" nodes:
				for (int i = 0; i < Math.min((CacheSizeLimit - newCacheSizeLimit), CacheHitOrder.size()); i++)
				{
					Cache.remove(CacheHitOrder.get(0));
					CacheHitOrder.remove(0);
				}
			}

			CacheSizeLimit = newCacheSizeLimit;
		}

		static public int getCacheSizeLimit()
		{
			return CacheSizeLimit;
		}

		/**
		 * Under Windows, the temporary files remain in place - they can't be deleted because they are locked.
		 * @throws Exception
		 */
		static public void cleanupTemporaryFiles() throws Exception
		{
			//	Try this: First empty the cache to remove any links to the temporary files. Doesn't work!
			Cache.clear();
			System.gc();

			//	This doesn't work under Windows - the temporary files remain in place:
			for (String s: TempPDFFileNames)		FileUtil.deleteOnExit(s);
			for (String s: TempPNGFileNames)		FileUtil.deleteOnExit(s);
		}

		/**
		 * Since cleaning-up temporary files on shutdown doesn't work under Windows,
		 * now I delete the whole temp-folder on startup. I delete as much as I can and swallow any exceptions.
		 * Do this in a background thread for not slowing down the initialization.
		 *(Deleting files may take a while because he retries several times on failure)
		 */
		static public void cleanupTemporaryFolder()
		{
			Logger.debug("Cleaning up temporary folder for file preview: " + FileUtil.getTempFolder() + "/FileConverter");

			new SwingWorker<Integer, Object>()
			{
				@Override
				public Integer doInBackground()
				{
					try
					{
						FileUtil.delete(FileUtil.getTempFolder() + "/FileConverter");
					}
					catch (FileUtilExceptionListException e){}		//	Ignore any exception

					return 0;
				}
			}.execute();

			Logger.debug("...done");
		}

		//	========	Static Private			=======================================================

		static private JTextArea createPreviewError(String text)
		{
			//	I need this JTextArea only in case of an error:
			JTextArea previewError = new JTextArea();
			previewError.setEditable(false);
			previewError.setLineWrap(true);
			previewError.setWrapStyleWord(true);
			previewError.setBackground(Color.YELLOW);
			previewError.setTabSize(4);
			previewError.setText(text);
			previewError.setCaretPosition(0);

			return previewError;
		}


		static private float zoomFactorToScaleProportionally(int imageWidth, int imageHeight, int displayWidth, int displayHeight)
		{
			return Math.min((float)displayWidth/imageWidth, (float)displayHeight/imageHeight);
		}

		//	===========================================================================================
		//	========	Inner Classes			=======================================================
		//	===========================================================================================

		static private enum FileType
		{
			Folder,
			Text, HTML, PDF, OOConvertable,
			GraphicsNative, GraphicsJAIConvertable, GraphicsJIMIConvertable, GraphicsImageMagickConvertable,
			Unknown;

			static private FileType check(NodeAbstract node)
			{
				if (node.isFolder())						return Folder;

				//	First PUID:
				String formatKey = node.getFormatKey();
				     if (formatKey == null)					;										//	Just continue
				else if (formatKey.equals("x-fmt/111"))		return Text;							//	Text
				else if (formatKey.equals("x-fmt/18"))		return Text;							//	CSV
				else if (formatKey.equals("x-fmt/22"))		return Text;							//	ASCII
				else if (formatKey.equals("x-fmt/283"))		return Text;							//	ASCII
				else if (formatKey.equals("fmt/96"))		return HTML;							//	HTML
				else if (formatKey.equals("fmt/97"))		return HTML;							//	HTML
				else if (formatKey.equals("fmt/98"))		return HTML;							//	HTML
				else if (formatKey.equals("fmt/99"))		return HTML;							//	HTML
				else if (formatKey.equals("fmt/100"))		return HTML;							//	HTML
				else if (formatKey.equals("fmt/102"))		return HTML;							//	XHTML
				else if (formatKey.equals("fmt/103"))		return HTML;							//	XHTML
				else if (formatKey.equals("fmt/471"))		return HTML;							//	HTML
				else if (formatKey.equals("fmt/14"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/15"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/16"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/17"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/18"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/19"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/20"))		return PDF;								//	PDF
				else if (formatKey.equals("fmt/95"))		return PDF;								//	PDF/A
				else if (formatKey.equals("fmt/354"))		return PDF;								//	PDF/A
				else if (formatKey.equals("fmt/144"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/145"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/146"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/147"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/148"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/157"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/158"))		return PDF;								//	PDF/X
				else if (formatKey.equals("fmt/41"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("fmt/42"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("fmt/43"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("fmt/44"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("x-fmt/390"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("x-fmt/391"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("x-fmt/398"))		return GraphicsNative;					//	JPG
				else if (formatKey.equals("fmt/3"))			return GraphicsNative;					//	GIF
				else if (formatKey.equals("fmt/4"))			return GraphicsNative;					//	GIF
				else if (formatKey.equals("fmt/11"))		return GraphicsNative;					//	PNG
				else if (formatKey.equals("fmt/12"))		return GraphicsNative;					//	PNG
				else if (formatKey.equals("fmt/13"))		return GraphicsNative;					//	PNG
				else if (formatKey.equals("fmt/114"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("fmt/115"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("fmt/116"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("fmt/117"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("fmt/118"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("fmt/119"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("x-fmt/270"))		return GraphicsJAIConvertable;			//	BMP
				else if (formatKey.equals("fmt/10"))		return GraphicsJAIConvertable;			//	TIFF
				else if (formatKey.equals("fmt/353"))		return GraphicsJAIConvertable;			//	TIFF
				else if (formatKey.equals("x-fmt/92"))		return GraphicsJIMIConvertable;			//	PSD
				else if (formatKey.equals("fmt/122"))		return GraphicsImageMagickConvertable;	//	EPS
				else if (formatKey.equals("fmt/123"))		return GraphicsImageMagickConvertable;	//	EPS
				else if (formatKey.equals("fmt/124"))		return GraphicsImageMagickConvertable;	//	EPS
				else if (formatKey.equals("fmt/45"))		return OOConvertable;					//	RTF
				else if (formatKey.equals("fmt/50"))		return OOConvertable;					//	RTF
				else if (formatKey.equals("fmt/51"))		return OOConvertable;					//	RTF
				else if (formatKey.equals("fmt/52"))		return OOConvertable;					//	RTF
				else if (formatKey.equals("fmt/53"))		return OOConvertable;					//	RTF
				else if (formatKey.equals("fmt/355"))		return OOConvertable;					//	RTF
				else if (formatKey.equals("fmt/37"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("fmt/38"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("fmt/39"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("fmt/40"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/129"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/131"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/273"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/274"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/275"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/276"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("x-fmt/473"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("fmt/412"))		return OOConvertable;					//	MS Word
				else if (formatKey.equals("fmt/42"))		return OOConvertable;					//	Wordperfect
				else if (formatKey.equals("fmt/43"))		return OOConvertable;					//	Wordperfect
				else if (formatKey.equals("fmt/44"))		return OOConvertable;					//	Wordperfect
				else if (formatKey.equals("fmt/55"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/56"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/57"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/59"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/60"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/61"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/62"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/172"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/173"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/174"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/175"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/176"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/177"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/178"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/214"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/445"))		return OOConvertable;					//	MS Excel
				else if (formatKey.equals("fmt/125"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/126"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("x-fmt/88"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/179"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/180"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/181"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/182"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/215"))		return OOConvertable;					//	MS Powerpoint
				else if (formatKey.equals("fmt/111"))		return OOConvertable;					//	MS OFfice
				else if (formatKey.equals("fmt/189"))		return OOConvertable;					//	MS Open Office

				//	Then check MIME Type:
				String mimeType = node.getMimeType();
				     if (mimeType == null)					;										//	Just continue
				else if (mimeType.equals("text/plain"))		return Text;							//	Text
				else if (mimeType.equals("text/xml"))		return Text;							//	XML
				else if (mimeType.equals("text/html"))		return HTML;							//	HTML
				else if (mimeType.equals("application/xhtml+xml"))
															return HTML;							//	XHTML
				else if (mimeType.equals("application/pdf"))return PDF;								//	PDF
				else if (mimeType.equals("image/jpeg"))		return GraphicsNative;					//	JPG
				else if (mimeType.equals("image/gif"))		return GraphicsNative;					//	GIF
				else if (mimeType.equals("image/png"))		return GraphicsNative;					//	PNG
				else if (mimeType.equals("application/postscript"))
															return GraphicsImageMagickConvertable;	//	EPS
				else if (mimeType.equals("application/rtf"))return OOConvertable;					//	RTF

				//	Finally check file extension:
				String extension = FileUtil.asFileNameExtension(node.getLabel());
				if (extension != null)
				{
					extension = extension.toLowerCase();
					     if (extension.equals("txt"))		return Text;							//	Text
					else if (extension.equals("xml"))		return Text;							//	XML
					else if (extension.equals("properties"))return Text;							//	Properties
					else if (extension.equals("props"))		return Text;							//	Properties
					else if (extension.equals("config"))	return Text;							//	Configuration
					else if (extension.equals("conf"))		return Text;							//	Configuration
					else if (extension.equals("log"))		return Text;							//	Log
					else if (extension.equals("html"))		return HTML;							//	HTML
					else if (extension.equals("htm"))		return HTML;							//	HTML
					else if (extension.equals("pdf"))		return PDF;								//	PDF
					else if (extension.equals("gif"))		return GraphicsNative;					//	GIF
					else if (extension.equals("png"))		return GraphicsNative;					//	PNG
					else if (extension.equals("jpeg"))		return GraphicsNative;					//	JPG
					else if (extension.equals("jpg"))		return GraphicsNative;					//	JPG
					else if (extension.equals("bmp"))		return GraphicsJAIConvertable;			//	BMP
					else if (extension.equals("tiff"))		return GraphicsJAIConvertable;			//	TIF
					else if (extension.equals("tif"))		return GraphicsJAIConvertable;			//	TIF
					else if (extension.equals("psd"))		return GraphicsJIMIConvertable;			//	PSD
					else if (extension.equals("eps"))		return GraphicsImageMagickConvertable;	//	EPS
					else if (extension.equals("rtf"))		return OOConvertable;					//	RTF
					else if (extension.equals("doc"))		return OOConvertable;					//	MS Word
					else if (extension.equals("docx"))		return OOConvertable;					//	MS X Word
					else if (extension.equals("xls"))		return OOConvertable;					//	MS Excel
					else if (extension.equals("xlsx"))		return OOConvertable;					//	MS X Excel
					else if (extension.equals("ppt"))		return OOConvertable;					//	MS PowerPoint
					else if (extension.equals("pptx"))		return OOConvertable;					//	MS X PowerPoint
				}

				return Unknown;
			}
		}


		/**
		 * The coding of this class follows closely the example
		 * ViewerComponentExample.java which is distributed with the ICEpdf
		 * library.
		 *
		 * @author andreas
		 *
		 */
		private class PDFPreviewer extends JPanel
		{
			private PDFPreviewer(FilePreviewPanel parent, String fileName)
			{
				super(new BorderLayout());

				SwingController controller = new SwingController();
				controller.setIsEmbeddedComponent(true);

				DocuteamPackerSwingViewBuilder factory = new DocuteamPackerSwingViewBuilder(controller);

				JToolBar toolBar = factory.buildCompleteToolBar(true);

				if (toolBar != null)
					this.add(toolBar, BorderLayout.NORTH);

				this.add(controller.getDocumentViewController().getViewContainer(), BorderLayout.CENTER);

				JPanel statusPanel = factory.buildStatusPanel();
				if (statusPanel != null)
					this.add(statusPanel, BorderLayout.SOUTH);

				controller.openDocument(fileName);
				// ToDo This somehow doesn't have the intended effect yet:
				controller.setPageFitMode(org.icepdf.core.views.DocumentViewController.PAGE_FIT_ACTUAL_SIZE, true);
			}

			/**
			 * The resason for using a subclass is to be able to use the
			 * sometimes private methods of its superclass.
			 *
			 * @author andreas
			 *
			 */
			protected class DocuteamPackerSwingViewBuilder extends SwingViewBuilder {

				public DocuteamPackerSwingViewBuilder(SwingController c) {
					super(c);
				}

				/**
				 * This reduces the toolbar to the necessary components for the preview.
				 */
				@Override
				public JToolBar buildCompleteToolBar(boolean embeddableComponent) {
					JToolBar toolbar = new JToolBar();
					this.commonToolBarSetup(toolbar, true);

					// Build the main set of toolbars based on the property file configuration
					this.addToToolBar(toolbar, this.buildPageNavigationToolBar());
					this.addToToolBar(toolbar, this.buildZoomToolBar());
					this.addToToolBar(toolbar, this.buildFitToolBar());

					// Set the toolbar back to null if no components were added
					// The result of this will properly disable the necessary menu items for controlling the toolbar
					if (toolbar.getComponentCount() == 0) {
						toolbar = null;
					}

					if ((this.viewerController != null) && (toolbar != null))
						this.viewerController.setCompleteToolBar(toolbar);

					return toolbar;
				}
			}

		}


		private class GraphicsPreviewer extends JLabel
		{
			private FilePreviewPanel	parent;

			private ImageIcon			previewGraphicsImageIconOriginalSize;


			private GraphicsPreviewer(FilePreviewPanel parent, ImageIcon imageIcon)
			{
				super();

				this.parent = parent;

				this.setVerticalAlignment(SwingConstants.CENTER);
				this.setHorizontalAlignment(SwingConstants.CENTER);
				this.addComponentListener(new ComponentAdapter() {
						@Override public void componentResized(ComponentEvent e) { GraphicsPreviewer.this.wasResized(); }});

				this.previewGraphicsImageIconOriginalSize = imageIcon;
			}


			private void wasResized()
			{
				this.setIcon(this.scaleProportionallyTo(this.previewGraphicsImageIconOriginalSize, this.parent.getWidth(), this.parent.getHeight()));
				this.repaint();
			}


			private ImageIcon scaleProportionallyTo(ImageIcon originalImageIcon, int displayWidth, int displayHeight)
			{
				int imageWidth = originalImageIcon.getIconWidth();
				int imageHeight = originalImageIcon.getIconHeight();

				float zoomFactor = zoomFactorToScaleProportionally(imageWidth, imageHeight, displayWidth, displayHeight);
				if (zoomFactor >= 1)		return originalImageIcon;

				int zoomedImageWidth = (int)(zoomFactor * imageWidth);
				int zoomedImageHeight = (int)(zoomFactor * imageHeight);

				return new ImageIcon(originalImageIcon.getImage().getScaledInstance(zoomedImageWidth, zoomedImageHeight, Image.SCALE_FAST));
			}
		}

	}
}
