package ch.docuteam.docutools.admin;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.*;
import java.net.URL;
import java.net.URLDecoder;
import java.util.*;

public class PackageStatistics
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Class Initialization	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private String						name;
	private int							totalFields = 0;
	private int							totalConstructors = 0;
	private int							totalMethods = 0;

	private List<ClassStatistics>		classesStatistics = new Vector<ClassStatistics>(100);

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	static public void main(String... args) throws ClassNotFoundException, IOException
	{
		if (args.length != 2)
		{
			System.out.println("1st parameter (mandatory) must be the package name to probe.");
			System.out.println("2nd parameter (mandatory) is 0 for no output, 1 for brief, 2 for medium, 3 for full.");
			return;
		}

		int detailLevel;
		try
		{
			detailLevel = new Integer(args[1]);
		}
		catch (NumberFormatException e)
		{
			System.out.println("Bad number: " + args[1]);
			return;
		}

		PackageStatistics ps = new PackageStatistics(args[0]);

		switch (detailLevel)
		{
			case 0:		break;
			case 1:		System.out.println(ps.toStringBrief());
						break;
			case 2:		System.out.println(ps.toStringMedium());
						break;
			case 3:		System.out.println(ps.toStringFull());
						break;
			default:	throw new IllegalArgumentException("Invalid 2nd parameter: " + detailLevel);
		}
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	public PackageStatistics(String name) throws ClassNotFoundException, IOException
	{
		this.name = name;

		for (Class<?> c: getClassesInPackage(name))
		{
			try
			{
				ClassStatistics cs = new ClassStatistics(c);
				this.classesStatistics.add(cs);
				this.totalFields += cs.getFieldsStatistics().size();
				this.totalConstructors += cs.getConstructorsStatistics().size();
				this.totalMethods += cs.getMethodsStatistics().size();
			}
			catch (ClassIsAnonymousException ex){	/*	Ignore anonymous classes */	}
		}
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	/**
	 * Scans all classes accessible from the context class loader which belong to the given package and subpackages.
	 *
	 * @param packageName The base package
	 * @return The classes
	 * @throws ClassNotFoundException
	 * @throws IOException
	 */
	static private List<Class<?>> getClassesInPackage(String packageName) throws ClassNotFoundException, IOException
	{
		String path = packageName.replace('.', '/');

		List<Class<?>> classes = new ArrayList<Class<?>>();
		for (Enumeration<URL> resources = ClassLoader.getSystemClassLoader().getResources(path); resources.hasMoreElements();)
		{
			classes.addAll(findClasses(new File(URLDecoder.decode(resources.nextElement().getFile(), "UTF-8")), packageName));
		}

		return classes;
	}

	/**
	 * Recursive method used to find all classes in a given directory and all its subdirs.
	 *
	 * @param directory The base directory
	 * @param packageName The package name for classes found inside the base directory
	 * @return The classes
	 * @throws ClassNotFoundException
	 */
	static private List<Class<?>> findClasses(File directory, String packageName) throws ClassNotFoundException
	{
		List<Class<?>> classes = new Vector<Class<?>>();
		if (!directory.exists())		return classes;

		for (File file: directory.listFiles())
		{
			String fileName = file.getName();
			if (file.isDirectory())
			{
				classes.addAll(findClasses(file, packageName + "." + fileName));		//	Recursion!
			}
			else if (fileName.endsWith(".class"))
			{
				try
				{
					classes.add(Class.forName(packageName + '.' + fileName.substring(0, fileName.length() - 6)));
				}
				catch (Throwable ex)
				{
					//	If the class can't be instanciated: ignore it and continue
				}
			}
		}
		return classes;
	}

	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public String getName()
	{
		return this.name;
	}

	public List<ClassStatistics> getClassesStatistics()
	{
		return this.classesStatistics;
	}

	public int getTotalFields()
	{
		return this.totalFields;
	}

	public int getTotalConstructors()
	{
		return this.totalConstructors;
	}

	public int getTotalMethods()
	{
		return this.totalMethods;
	}


	public String toStringFull()
	{
		StringBuilder string = new StringBuilder(this.toString());

		for (ClassStatistics cs: this.classesStatistics)		string.append("\n\t").append(cs.toStringFull());
		string.append("\n\tClasses     : ").append(this.classesStatistics.size());
		string.append("\n\tFields      : ").append(this.totalFields);
		string.append("\n\tConstructors: ").append(this.totalConstructors);
		string.append("\n\tMethods     : ").append(this.totalMethods);

		return string.toString();
	}


	public String toStringMedium()
	{
		StringBuilder string = new StringBuilder(this.toString());

		for (ClassStatistics cs: this.classesStatistics)		string.append("\n\t").append(cs.toStringBrief());
		string.append("\n\tClasses     : ").append(this.classesStatistics.size());
		string.append("\n\tFields      : ").append(this.totalFields);
		string.append("\n\tConstructors: ").append(this.totalConstructors);
		string.append("\n\tMethods     : ").append(this.totalMethods);

		return string.toString();
	}


	public String toStringBrief()
	{
		StringBuilder string = new StringBuilder(this.toString());

		string.append("\n\tClasses     : ").append(this.classesStatistics.size());
		string.append("\n\tFields      : ").append(this.totalFields);
		string.append("\n\tConstructors: ").append(this.totalConstructors);
		string.append("\n\tMethods     : ").append(this.totalMethods);

		return string.toString();
	}
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return "Package: " + this.name;
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	static public class ClassStatistics
	{
		private String							name;

		private List<FieldStatistics>			fieldsStatistics = new Vector<FieldStatistics>(10);
		private List<ConstructorStatistics>		constructorsStatistics = new Vector<ConstructorStatistics>(10);
		private List<MethodStatistics>			methodsStatistics = new Vector<MethodStatistics>(10);


		private ClassStatistics(Class<?> cls) throws ClassIsAnonymousException
		{
			if (cls.isAnonymousClass())		throw new ClassIsAnonymousException(cls);

			this.name = cls.getCanonicalName();

			for (Field f: cls.getDeclaredFields())					this.fieldsStatistics.add(new FieldStatistics(f));
			for (Constructor<?> c: cls.getDeclaredConstructors())	this.constructorsStatistics.add(new ConstructorStatistics(c));
			for (Method m: cls.getDeclaredMethods())				this.methodsStatistics.add(new MethodStatistics(m));

			//	Get file size and row count?
		}


		public String getName()
		{
			return this.name;
		}

		public List<MethodStatistics> getMethodsStatistics()
		{
			return this.methodsStatistics;
		}

		public List<ConstructorStatistics> getConstructorsStatistics()
		{
			return this.constructorsStatistics;
		}

		public List<FieldStatistics> getFieldsStatistics()
		{
			return this.fieldsStatistics;
		}


		public String toStringFull()
		{
			StringBuilder string = new StringBuilder(this.toString());

			for (FieldStatistics fs: this.fieldsStatistics)					string.append("\n\t\t").append(fs.toString());
			for (ConstructorStatistics cs: this.constructorsStatistics)		string.append("\n\t\t").append(cs.toString());
			for (MethodStatistics ms: this.methodsStatistics)				string.append("\n\t\t").append(ms.toString());
			string.append("\n\t\tFields      : ").append(this.fieldsStatistics.size());
			string.append("\n\t\tConstructors: ").append(this.constructorsStatistics.size());
			string.append("\n\t\tMethods     : ").append(this.methodsStatistics.size());

			return string.toString();
		}


		public String toStringBrief()
		{
			StringBuilder string = new StringBuilder(this.toString());

			string.append("\n\t\tFields      : ").append(this.fieldsStatistics.size());
			string.append("\n\t\tConstructors: ").append(this.constructorsStatistics.size());
			string.append("\n\t\tMethods     : ").append(this.methodsStatistics.size());

			return string.toString();
		}


		@Override
		public String toString()
		{
			return "Class: " + this.name;
		}
	}


	static public class FieldStatistics
	{
		private String		name;

		private FieldStatistics(Field field)
		{
//				this.name = field.toString();
			this.name = field.getName();
		}


		public String getName()
		{
			return this.name;
		}

		@Override
		public String toString()
		{
			return "Field: " + this.name;
		}
	}


	static public class ConstructorStatistics
	{
		private String		name;

		private ConstructorStatistics(Constructor<?> constructor)
		{
//				this.name = constructor.toString();
			this.name = constructor.getName();
		}


		public String getName()
		{
			return this.name;
		}

		@Override
		public String toString()
		{
			return "Constructor: " + this.name;
		}
	}


	static public class MethodStatistics
	{
		private String		name;

		private MethodStatistics(Method method)
		{
//				this.name = method.toString();
			this.name = method.getName();
		}


		public String getName()
		{
			return this.name;
		}

		@Override
		public String toString()
		{
			return "Method: " + this.name;
		}
	}


	static private class ClassIsAnonymousException extends Exception
	{
		private ClassIsAnonymousException(Class<?> cls)
		{
			super("Can't create statistics for anonymous class: " + cls.getName());
		}
	}

}
