/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.db;


import java.sql.*;

import ch.docuteam.docutools.out.Logger;

/**
 * @author denis
 *
 */
public abstract class MySQLJDBCConnection
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//		DB Property File Keys:
	
	static public final String		PropertyNameDBHost = "DBHost";
	static public final String		PropertyNameDBPort = "DBPort";
	static public final String		PropertyNameDBName = "DBName";
	static public final String		PropertyNameDBUserName = "DBUserName";
	static public final String		PropertyNameDBUserPassword = "DBUserPassword";

	//		DB SSL Property File Keys:
	
	static public final String		PropertyNameSSLKeyStore = "javax.net.ssl.keyStore";
	static public final String		PropertyNameSSLKeyStorePassword = "javax.net.ssl.keyStorePassword";
	static public final String		PropertyNameSSLTrustStore = "javax.net.ssl.trustStore";
	static public final String		PropertyNameSSLTrustStorePassword = "javax.net.ssl.trustStorePassword";

	//	========	Static Final Private	=======================================================

	static private final String		DBConnectionPrefix = "jdbc:mysql://";
	static private final String		DBConnectionSSLPostfix = "?verifyServerCertificate=true&useSSL=true&requireSSL=true";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private String			ConnectionString;
	static private String			UserName;
	static private String			Password;
	
	static private Connection		Connection;
	
	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	//	--------		Factories			-------------------------------------------------------
	
	/**
	 * 	Create the JDBCConnection
	 */
	static public Connection create(String dbHost, String dbPort, String dbName, String userName, String password) throws SQLException
	{
		ConnectionString = DBConnectionPrefix + dbHost + ":" + dbPort + "/" + dbName;
		UserName = userName;
		Password = password;

		return create();
	}

	//	--------		SSL Factories		-------------------------------------------------------
	
	/**
	 * 	Create the JDBCConnection using a SSL
	 */
	static public Connection createUsingSSL(String dbHost, String dbPort, String dbName, String userName, String password, String sslKeyStore, String sslKeyStorePassword, String sslTrustStore, String sslTrustStorePassword) throws SQLException
	{
		System.setProperty(PropertyNameSSLKeyStore, sslKeyStore);
		System.setProperty(PropertyNameSSLKeyStorePassword, sslKeyStorePassword);
		System.setProperty(PropertyNameSSLTrustStore, sslTrustStore);
		System.setProperty(PropertyNameSSLTrustStorePassword, sslTrustStorePassword);

		ConnectionString = DBConnectionPrefix + dbHost + ":" + dbPort + "/" + dbName + DBConnectionSSLPostfix;
		UserName = userName;
		Password = password;

		return create();
	}
	
	//	--------		Accessing			-------------------------------------------------------

	static public Connection getConnection()
	{
		return Connection;
	}
	
	//	--------		Queries				-------------------------------------------------------
	
	static public ResultSet executeQuery(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.info("\t" + query);
		return Connection.createStatement().executeQuery(query);
	}
	
	
	static public int executeUpdate(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.info("\t" + query);
		int rowCount = Connection.createStatement().executeUpdate(query);
		Logger.info("\t-> " + rowCount);
		return rowCount;
	}
	
	
	static public ResultSet executeInsert(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.info("\t" + query);
		Statement statement = Connection.createStatement();
		int rowCount = statement.executeUpdate(query, new String[] {"id"});
		Logger.info("\t-> " + rowCount);
		return statement.getGeneratedKeys();
	}
	
	
	static public PreparedStatement prepareStatement(String query) throws SQLException
	{
		reOpenIfNecessary();

		Logger.info("\t" + query);
		return Connection.prepareStatement(query, new String[] {"id"});
	}
	
	
	//	--------		Closing				-------------------------------------------------------
	
	/**
	 * Close the connection
	 * @throws SQLException
	 */
	static public void close() throws SQLException
	{
		Logger.debug("Closing");
		if (Connection != null)		Connection.close();
	}

	//	========	Static Private			=======================================================

	static private Connection create() throws SQLException
	{
		Logger.debug("Creating: " + ConnectionString);
		return Connection = DriverManager.getConnection(ConnectionString, UserName, Password);
	}
	

	static private Connection reOpenIfNecessary() throws SQLException
	{
		if ((Connection == null) || !Connection.isValid(0))		create();
		return Connection;
	}

}
