/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.string;

import java.util.*;

/**
 * @author denis
 *
 */
public abstract class StringUtil
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	static public boolean isNumeric(String s)
	{
		if (s == null || s.isEmpty())		return false;

		for (int i = 0; i < s.length(); i++)		if (!Character.isDigit(s.charAt(i)))			return false;

		return true;
	}


	static public boolean isAlphaNumeric(String s)
	{
		if (s == null || s.isEmpty())		return false;

		for (int i = 0; i < s.length(); i++)		if (!Character.isLetterOrDigit(s.charAt(i)))	return false;

		return true;
	}


	static public boolean isAlpha(String s)
	{
		if (s == null || s.isEmpty())		return false;

		for (int i = 0; i < s.length(); i++)		if (!Character.isLetter(s.charAt(i)))			return false;

		return true;
	}


	/**
	 * Return a String with all characters except digits filtered away.
	 * @param s: input string
	 * @return result string
	 */
	static public String onlyDigits(String s)
	{
		StringBuffer result = new StringBuffer();

		for (int i = 0; i < s.length(); i++)
		{
			char c = s.charAt(i);
			if (Character.isDigit(c))		result.append(c);
		}

		return result.toString();
	}


	/**
	 * Return a String with all characters except alphanumeric and white-spaces filtered away.
	 * @param s: input string
	 * @return result string
	 */
	static public String onlyAlphasDigitsWhitespaces(String s)
	{
		StringBuffer result = new StringBuffer();

		for (int i = 0; i < s.length(); i++)
		{
			char c = s.charAt(i);
			if (Character.isLetterOrDigit(c) || Character.isWhitespace(c))		result.append(c);
		}

		return result.toString();
	}


	/**
	 * Return the index position of the occurence's occurrence of the search string within the string string. If the occurrence is negative, search backwards.
	 * @param string: The string to search in
	 * @param search: The string to search for
	 * @param occurrence The occurrence. If negative, search backwards. If 0, return -1
	 * @return The index, or -1 if not found.
	 */
	static public int indexOf(String string, String search, int occurrence)
	{
		if (occurrence == 0)						return -1;
		if (search.isEmpty())						return -1;
		if (string.isEmpty())						return -1;
		if (string.length() < search.length())		return -1;
		if (string.equals(search))					return 0;

		int count = 0;
		int i;

		if (occurrence > 0)
		{
			i = -1;
			do
			{
				i = string.indexOf(search, ++i);
				if (i != -1)	count++;
			}
			while ((i != -1) && (count < occurrence));
		}
		else
		{
			i = string.length() + 1;
			do
			{
				i = string.lastIndexOf(search, --i);
				if (i != -1)	count++;
			}
			while ((i != -1) && (count < -occurrence));
		}

		return i;
	}


	/**
	 * Return the number of occurrences of the search string in the string.
	 * @param string: The string to search in
	 * @param search: The string to search for
	 * @return The count, or 0 if not found.
	 */
	static public int occurrencesOf(String string, String search)
	{
		if (search.isEmpty())					return 0;
		if (string.isEmpty())					return 0;
		if (string.length() < search.length())	return 0;

		int count = 0;

		for (int i = 0; i <= (string.length() - search.length()); i++)
		{
			if (string.substring(i, i + search.length()).equals(search))	count++;
		}

		return count;
	}


	/**
	 *
	 */
	static public String concatSeparatedBy(Iterable<String> elements, String separator)
	{
		StringBuilder stringBuilder = new StringBuilder();

		boolean isFirst = true;
		for (String s: elements)
		{
			if (isFirst)	isFirst = false;
			else			stringBuilder.append(separator);

			stringBuilder.append(s);
		}

		return stringBuilder.toString();
	}


	static public String concatSeparatedBy(String[] elements, String separator)
	{
		return concatSeparatedBy(Arrays.asList(elements), separator);
	}


	/**
	 * Split the string into tokens, separated by a space character.
	 * Note: if the delimiter is not found in the string, the result contains exactly one entry, that is the complete string.
	 * @param string: the string to split
	 * @param delimiter: the token delimiter
	 * @return The List of tokens
	 */
	static public List<String> split(String string)
	{
		return split(string, " ");
	}


	/**
	 * Split the string into tokens, separated by a space character.
	 * Note: if the delimiter is not found in the string, the result contains exactly one entry, that is the complete string.
	 * @param string: the string to split
	 * @param delimiter: the token delimiter
	 * @return The List of tokens
	 */
	static public List<String> split(String string, String delimiter)
	{
		return split(string, delimiter, true);
	}


	/**
	 * Split the string into tokens, separated by the delimiter (excluding).
	 * Note: if the delimiter is not found in the string, the result contains exactly one entry, that is the complete string.
	 * @param string: the string to split
	 * @param delimiter: the token delimiter
	 * @return The List of tokens
	 */
	static public List<String> split(String string, String delimiter, boolean ignoreEmptyTokens)
	{
		List<String> tokens = new ArrayList<String>();

		if (delimiter.isEmpty())								return tokens;
		if (string.isEmpty() && ignoreEmptyTokens)				return tokens;

		int beginIndex = 0;
		int endIndex = 0;
		do
		{
			endIndex = string.indexOf(delimiter, beginIndex);
			if (endIndex != -1)
			{
				String token = string.substring(beginIndex, endIndex);
				if (!(token.isEmpty() && ignoreEmptyTokens))	tokens.add(token);
				beginIndex = endIndex + delimiter.length();
			}
		}
		while (endIndex != -1);

		String token = string.substring(beginIndex, string.length());
		if (!(token.isEmpty() && ignoreEmptyTokens))			tokens.add(token);

		return tokens;
	}


	/**
	 * Does a standard split, but takes into account quotes
	 * Standard delimiter is space, standard quote is double-quote.
	 */
	static public List<String> splitQuoted(String string)
	{
		return splitQuoted(string, "\"");
	}

	/**
	 * Does a standard split, but takes into account quotes
	 * Standard delimiter is space.
	 */
	static public List<String> splitQuoted(String string, String quote)
	{
		return splitQuoted(string, quote, " ");
	}

	/**
	 * Does a standard split, but takes into account quotes
	 */
	static public List<String> splitQuoted(String string, String quote, String delimiter)
	{
		if (!string.contains(quote))								return split(string, delimiter, true);							//	No quotes
		if (StringUtil.occurrencesOf(string, quote) % 2 == 1)		return split(string.replace(quote, ""), delimiter, true);		//	Odd number of quotes: ignore!

		List<String> tokens = new ArrayList<String>();
		List<String> quokens = split(string, quote, false);
		for (int i = 0; i < quokens.size(); i++)
		{
			if (i % 2 == 0)		tokens.addAll(split(quokens.get(i).trim(), delimiter, true));		//	Outside quotes: standard split
			else				tokens.add(quokens.get(i));											//	Within quotes: is ONE token
		}

		return tokens;
	}


	/**
	 * Return a String containing the first i characters of the string. If i is larger than the string, return the string itself.
	 * @param string
	 * @param i
	 * @return
	 */
	static public String first(String string, Integer i)
	{
		if (i >= string.length())		return string;

		return string.substring(0, i);
	}


	/**
	 * Return a String containing the last i characters of the string. If i is larger than the string, return the string itself.
	 * @param string
	 * @param i
	 * @return
	 */
	static public String last(String string, Integer i)
	{
		if (i >= string.length())		return string;

		return string.substring(string.length() - i);
	}



	/**
	 * The resulting string has a length of size characters. If the string s is shorter, add as many c's to the right. If the string is too long, cut the right part off.
	 * @param o
	 * @param c
	 * @param size
	 * @return
	 */
	static public String padUpRightWith(String s, Character c, int size)
	{
		StringBuilder stringBuilder = new StringBuilder(s);
		for (int i = 0; i < size; i++)	stringBuilder.append(c);

		return stringBuilder.substring(0, size);
	}


	/**
	 * The resulting string has a length of size characters. If the string s is shorter, add as many c's to the left. If the string is too long, cut the left part off.
	 * @param o
	 * @param c
	 * @param size
	 * @return
	 */
	static public String padUpLeftWith(String s, Character c, int size)
	{
		StringBuilder stringBuilder = new StringBuilder();
		for (int i = 0; i < size; i++)	stringBuilder.append(c);
		stringBuilder.append(s);

		return stringBuilder.substring(stringBuilder.length() - size);
	}


	/**
	 * Wrap the string so that every line (except the last one) is exactly wrapWidth characters long.
	 * @param string
	 * @param wrapWidth
	 * @return
	 */
	static public List<String> wrap(String string, int wrapWidth)
	{
		List<String> lines = new ArrayList<String>();
		int stringLength = string.length();

		for (int i = 0; i < stringLength; i += wrapWidth)
		{
			lines.add(string.substring(i, Math.min(i + wrapWidth, stringLength)));
		}

		return lines;
	}


	/**
	 * Wrap the string on word boundaries so that every line is maximally wrapWidth characters long.
	 * @param string
	 * @param wrapWidth
	 * @return
	 */
	static public List<String> wordWrap(String string, int wrapWidth)
	{
		List<String> lines = new ArrayList<String>();
		int stringLength = string.length();

		int startPos = 0;
		boolean done = false;
		do
		{
			int endPos = Math.min(startPos + wrapWidth, stringLength);
			if (endPos == stringLength)		done = true;
			else
			{
				int lastSpacePos = string.lastIndexOf(" ", endPos);
				if (lastSpacePos != -1)		endPos = lastSpacePos;
			}

			//	This trim() is in case there are multiple spaces at the end of a line:
			lines.add(string.substring(startPos, endPos).trim());
			startPos = endPos + 1;
		}
		while (!done);

		return lines;
	}


	/**
	 * Remove leading whitespace
	 */
	public static String trimLeft(String string)
	{
		return string.replaceAll("^\\s+","");
	}


	/**
	 * Remove trailing whitespace
	 */
	public static String trimRight(String string)
	{
		return string.replaceAll("\\s+$","");
	}


	/**
	 * Remove leading and trailing whitespace
	 */
	public static String trimLeftAndRight(String string)
	{
		return trimLeft(trimRight(string));
	}


	/**
	 * Replace multiple whitespaces between words with a single space
	 */
	public static String trimMid(String string)
	{
		return string.replaceAll("\\b\\s{2,}\\b"," ");
	}


	/**
	 * Replace all multiple whitespaces in source string with a single space, and remove all leading and trailing whitespaces
	 */
	public static String trimAll(String string)
	{
		return trimLeftAndRight(trimMid(string));
	}


	/**
	 * Increment the string s, meaning to increment the rightmost alphanumeric character:
	 * 1->2
	 * 9->10
	 * a->b
	 * z->za
	 * A->B
	 * Z->ZA
	 * If no alphanumeric character is present, add "1" to the end of the string:
	 * +++->+++1
	 * Special cases (overflow):
	 * 49->50
	 * 999->1000
	 * @param s The string to be incremented
	 * @return the incremented string
	 */
	public static String increment(String s)
	{
		//	Find the last character that can be incremented (a letter or digit):
		for (int i = s.length() - 1; i >= 0; i--)
		{
			if (Character.isLetterOrDigit(s.charAt(i)))
			{
				char c = s.charAt(i);

				//	Increment the character:
				if (c == '9')
				{
					//	In this case, extract the whole number and increment it:
					String numberString = "9";
					int j;
					for (j = i - 1; j >= 0; j--)
					{
						char digit = s.charAt(j);
						if (!Character.isDigit(digit))		break;

						numberString = digit + numberString;
					}

					long number = new Long(numberString).longValue();
					return s.substring(0, j + 1) + new Long(++number) + s.substring(i + 1);
				}
				else if (c == 'z')		return s.substring(0, i) + "za" + s.substring(i + 1);
				else if (c == 'Z')		return s.substring(0, i) + "ZA" + s.substring(i + 1);
				else					return s.substring(0, i) + new Character(++c).toString() + s.substring(i + 1);
			}
		}

		//	If none found: add a "1" to the end if the input string and return that:
		return s + "1";
	}

	//	========	Static Private			=======================================================

}
