/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.file;
/**
 *
 */

import java.io.*;
import java.util.HashMap;
import java.util.Map;

import ch.docuteam.docutools.os.EnvironmentVariable;
import ch.docuteam.docutools.out.Logger;


/**
 * @author denis
 *
 */
public abstract class PropertyFile
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static final private String				AssignmentString = "=";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static protected String					PropertyFileName = null;		//	This variable must be set to the appropriate value in the subclass (using the static initializer)
	static private Map<String, String>		Map = null;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	static public void initialize(String propsFileName)
	{
		//	Every call to initialize() has to make "tabula raza", in case that the initialization fails.
		PropertyFileName = null;
		Map = null;

		Logger.info("Reading file: " + propsFileName);

		BufferedReader reader = null;
		int lineNr = 0;
		String line;
		try
		{
			reader = new BufferedReader(new FileReader(propsFileName));
			Map = new HashMap<String, String>();

			do
			{
				++lineNr;
				line = reader.readLine();

				if (line == null)			break;			//	EOF reached

				line = line.trim();
				if (line.length() == 0)		continue;		//	Skip empty lines
				if (line.startsWith("#"))	continue;		//	Skip comment lines
				if (line.startsWith("//"))	continue;		//	Skip comment lines

				int indexOfAssignmentString = line.indexOf(AssignmentString);
				if (indexOfAssignmentString == -1)	throw new IllegalArgumentException("Error in row " + lineNr + ":'" + line + "': Assignment string '" + AssignmentString + "' missing");

				String key = line.substring(0, indexOfAssignmentString).trim();
				String value = line.substring(indexOfAssignmentString + 1).trim();

				//	Resolve possible environment variables in value:
				value = EnvironmentVariable.resolveIn(value);

				Logger.debug("Reading item(" + lineNr + "): " + key + AssignmentString + value);

				if (Map.containsKey(key))	Logger.warn("Duplicate key in file='" + propsFileName + "': " + key);
				Map.put(key, value);
			}
			while(line != null);
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			if (reader != null)		try { reader.close(); } catch (IOException e) { e.printStackTrace(); }
		}
	}

	//	========	Static Public			=======================================================

	static public String get(String key, String ifAbsent)
	{
		if (Map == null)		initialize();

		return Map.containsKey(key)? Map.get(key): ifAbsent;
	}

	static public String get(String key)
	{
		if (Map == null)		initialize();

		if (!Map.containsKey(key))
		{
			Logger.warn("Couldn't find value for key '" + key + "'");
			return "";
		}

		return Map.get(key);
	}

	//	========	Static Private			=======================================================

	static private void initialize()
	{
		initialize(PropertyFileName);
	}

}
