/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.gui;

import java.awt.Cursor;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Observable;
import java.util.Observer;

import javax.swing.*;

/**
 * @author denis
 *
 * The SmallPeskyMessageWindow is a window that can not be closed or moved by the user (that's why it's called pesky).
 * The only way to remove it is to close the parent window or quit the application. Or programatically, to send "close()" to the window.
 *
 * Opening and closing an instance of SmallPeskyMessageWindow has to happen within an instance of SwingWorker,
 * otherwise it will not be rendered correctly and remains empty.
 *
 * Example:
 * <code>
				new SwingWorker<Integer, Object>()
				{
					@Override public Integer doInBackground()
					{
						SmallPeskyMessageWindow w = SmallPeskyMessageWindow.open(f, "Ready");
						try
						{
							Thread.sleep(1000);
							w.setText("Steady");
							Thread.sleep(1000);
							w.setText("Go!");
							Thread.sleep(1000);
						}
						catch (InterruptedException ex){}
						w.close();

						return 0;
					}
				}.execute();
 * </code>
 */
public class SmallPeskyMessageWindow extends JDialog implements Observer
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private JLabel		label = new JLabel("", JLabel.CENTER);

	private boolean		isBlocking = false;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * @param args
	 */
	public static void main(String[] args)
	{
		final JFrame f = new JFrame();
		f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		JButton b = new JButton();
		b.addActionListener(new ActionListener()
		{ @Override public void actionPerformed(ActionEvent e)
			{
				new SwingWorker<Integer, Object>()
				{
					@Override public Integer doInBackground()
					{
						final SmallPeskyMessageWindow w = SmallPeskyMessageWindow.openBlocking(f, "Ready");
//						final SmallPeskyMessageWindow w = SmallPeskyMessageWindow.open(f, "Ready");

						try
						{
							Thread.sleep(1000);
							w.setText("Steady");
							Thread.sleep(1000);
							w.setText("Go!");
							Thread.sleep(1000);
						}
						catch (InterruptedException ex){}
						finally
						{
							w.close();
						}

						return 0;
					}
				}.execute();
			}
		});
		f.add(b);
		f.setSize(400, 200);
		f.setLocationRelativeTo(null);
		f.setVisible(true);
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	private SmallPeskyMessageWindow(Window parent, String message, boolean isBlocking)
	{
		super(parent);

		this.isBlocking = isBlocking;
		if (this.isBlocking && parent != null)		parent.setEnabled(false);

		this.setUndecorated(true);
		this.label.setText(message);
		this.add(this.label);
		this.setSize(300, 100);
		this.setLocationRelativeTo(parent);
	}

	//	========	Static Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------

	/**
	 * Open a smallPeskyMessageWindow over the parent window. The parent window is NOT blocked, so the user can still work with it.
	 * @param parent
	 * @param message
	 * @return
	 */
	static public SmallPeskyMessageWindow open(Window parent, String message)
	{
		SmallPeskyMessageWindow w = new SmallPeskyMessageWindow(parent, message, false);
		w.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		w.setVisible(true);

		return w;
	}

	/**
	 * Use "openBlocking(...)" when the parent window shall be blocked as long as the smallPeskyMessageWindow is open.
	 * @param parent
	 * @param message
	 * @return
	 */
	static public SmallPeskyMessageWindow openBlocking(Window parent, String message)
	{
		SmallPeskyMessageWindow w = new SmallPeskyMessageWindow(parent, message, true);
		w.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		w.setVisible(true);

		return w;
	}

	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	public void setText(String message)
	{
		this.label.setText(message);
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------

	/* (non-Javadoc)
	 * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
	 */
	@Override
	public void update(Observable obs, Object obj)
	{
		this.setText(obj.toString());
	}

	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------

	public void close()
	{
		this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
		this.setVisible(false);
		this.dispose();

		if (this.isBlocking && this.getParent() != null)		this.getParent().setEnabled(true);
	}

	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
