/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.os;

import java.util.Map;

import ch.docuteam.docutools.out.Logger;

public abstract class EnvironmentVariable
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String			HomeDir = ".";

	static private final String			EnvVarStartString = "${";
	static private final String			EnvVarEndString = "}";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	/**
	 * Returns an unmodifiable string map view of the current system environment.
	 * The environment is a system-dependent mapping from names to
	 * values which is passed from parent to child processes.
	 *
	 * <p>If the system does not support environment variables, an
	 * empty map is returned.
	 */
	static public Map<String, String> getAll()
	{
		return System.getenv();
	}


	/**
	 * Return the value of the environment variable 'name', or null if undefined.
	 */
	static public String get(String name)
	{
		return get(name, null);
	}


	/**
	 * Return the value of the environment variable 'name', or the 'ifUndefined' String if undefined.
	 */
	static public String get(String name, String ifUndefined)
	{
		String value = System.getenv(name);
		if (value == null)
		{
			if (ifUndefined == null)		Logger.warn("Undefined Environment Variable: " + name);
			value = ifUndefined;
		}

		Logger.debug("Retrieving Environment Variable: '" + name + "' with default: '" + ifUndefined + "' returning: '" + value + "'");

		return value;
	}


	/**
	 * Return the value of the environment variable 'name', or the home path "." if undefined.
	 */
	static public String getAsFolder(String name)
	{
		return get(name, HomeDir);
	}



	/**
	 * The input string may contain one or more environment variables (embedded between '${' and '}'). If this is the case, resolve them.
	 * @param string
	 * @return the string itself with possible environment variables replaced by their content. The content may be empty.
	 */
	static public String resolveIn(String string)
	{
		if ((string == null)
		|| !(string.contains(EnvVarStartString) && string.contains(EnvVarEndString)))		return string;

		int startIndex = 0;
		do
		{
			int envVarStartIndex = string.indexOf(EnvVarStartString, startIndex);
			if (envVarStartIndex == -1)		break;
			int envVarEndIndex = string.indexOf(EnvVarEndString, envVarStartIndex);
			if (envVarEndIndex == -1)		break;

			String envVarName = string.substring(envVarStartIndex + EnvVarStartString.length(), envVarEndIndex);
			String envVarContent = get(envVarName);
			if (envVarContent == null)		envVarContent = "";
			string = string.replace(string.substring(envVarStartIndex, envVarEndIndex + EnvVarEndString.length()), envVarContent);

			startIndex = envVarStartIndex + envVarContent.length();
		}
		while (true);

		return string;
	}

	//	========	Static Private			=======================================================

}
