/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.out;

import java.awt.Desktop;
import java.net.URI;
import java.util.Properties;

import javax.mail.*;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;


/**
 * @author Andreas Nef, Docuteam GmbH, 2010
 * @author Denis Lemberger, Docuteam GmbH, 2010
 */

public abstract class MailSender
{
	static public boolean sendMessageSMTP(String subject, String body, final Properties props)
	{
		return sendMessageSMTP(props.getProperty("mail.receiver"), subject, body, props);
	}


	static public boolean sendMessageSMTP(String receiver, String subject, String body, final Properties props)
	{
		try
		{
			Session session = null;
			if (Boolean.parseBoolean(props.getProperty("mail.smtp.auth"))) {
				session = Session.getDefaultInstance(props,
					new javax.mail.Authenticator()
					{
						@Override
						protected PasswordAuthentication getPasswordAuthentication()
						{
							return new PasswordAuthentication(props.getProperty("mail.smtp.user"), props.getProperty("mail.smtp.pwd"));
						}
					});
			}
			else
			{
				session = Session.getDefaultInstance(props);
			}
			MimeMessage message = new MimeMessage(session);
			message.setSender(new InternetAddress(props.getProperty("mail.smtp.user")));
			message.addRecipient(Message.RecipientType.TO, new InternetAddress(receiver));
			message.setSubject(subject);
//			message.setContent(body, "text/plain");		//	This doesn't work with umlauts in the body.
//			message.setText(body);
			message.setText(body, "UTF-8");				//	Does this work with umlauts in the body?
			Transport.send(message, message.getAllRecipients());
		}
		catch (NoSuchProviderException e)
		{
			Logger.warn("SMTP-Transfer did not succeed.", e);
			return false;
		}
		catch (MessagingException e)
		{
			Logger.warn("SMTP-Server " + props.getProperty("mail.smtp.host") + " ist not available", e);
			return false;
		}

		return true;
	}


	static public boolean sendMessageLocalClient(String subject, String body, Properties props)
	{
		return sendMessageLocalClient(props.getProperty("mail.receiver"), subject, body, props);
	}


	static public boolean sendMessageLocalClient(String receiver, String subject, String body, Properties props)
	{
		Logger.info("Opening local mail editor with mail template...");

		try
		{
			Desktop.getDesktop().mail(new URI("mailto", receiver + "?SUBJECT=" + subject + "&BODY=" + body, null));
		}
		catch (java.lang.Exception e)
		{
			Logger.warn("Mail could not be sent.", e);
			return false;
		}

		return true;
	}

}