/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docupack.gui;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.List;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import ch.docuteam.docudarc.mets.Document;
import ch.docuteam.docudarc.mets.structmap.NodeAbstract;
import ch.docuteam.docutools.gui.GridBagPanel;
import ch.docuteam.docutools.gui.SmallPeskyMessageWindow;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.string.Pattern;
import ch.docuteam.docutools.translations.I18N;

/**
 * @author denis
 *
 */
public class SearchView extends JFrame
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String		ScreenTitle = I18N.translate("TitleSearchInWorkspace");
	static private Dimension		StartScreenSize = new Dimension(600, 400);

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private SearchView		SearchView = null;

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	private LauncherView			launcherView;

	private JTable					searchResultTable;
	private JLabel					rowCountLabel;

	private JTextField				fileNameTextField;
	private JTextField				metadataTextField;
	private JTextField				fileContentTextField;

	private Action					clearAction;
	private Action					searchAction;
	private Action					pickAction;

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	private SearchView(LauncherView launcherView)
	{
		super(ScreenTitle);
		this.setIconImage(Toolkit.getDefaultToolkit().getImage("./resources/images/DocuteamPacker.png"));

		this.launcherView = launcherView;

		this.searchResultTable = new JTable(new SearchTableModel());
		this.searchResultTable.setAutoCreateRowSorter(true);
		this.searchResultTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		this.searchResultTable.getSelectionModel().addListSelectionListener(
				new ListSelectionListener() { @Override public void valueChanged(ListSelectionEvent e){ if (!e.getValueIsAdjusting()) SearchView.this.tableSelectionChanged(); }});
		this.searchResultTable.addMouseListener(
				new MouseAdapter() { @Override public void mousePressed(MouseEvent e){ if (e.getClickCount() == 2) SearchView.this.pickButtonClicked(); }});
		this.searchResultTable.getColumnModel().getColumn(0).setPreferredWidth(50);
		this.searchResultTable.getColumnModel().getColumn(1).setPreferredWidth(300);

		this.rowCountLabel = new JLabel();

		this.fileNameTextField = new JTextField();
		this.fileNameTextField.addActionListener(
				new ActionListener(){ @Override public void actionPerformed(ActionEvent e){ SearchView.this.searchButtonClicked(); }});
		this.fileNameTextField.setToolTipText(I18N.translate("ToolTipSearchInWorkspaceFileNameTextField"));

		this.metadataTextField = new JTextField();
		this.metadataTextField.addActionListener(
				new ActionListener(){ @Override public void actionPerformed(ActionEvent e){ SearchView.this.searchButtonClicked(); }});
		this.metadataTextField.setToolTipText(I18N.translate("ToolTipSearchInWorkspaceMetadataTextField"));

		this.fileContentTextField = new JTextField();
		this.fileContentTextField.addActionListener(
				new ActionListener(){ @Override public void actionPerformed(ActionEvent e){ SearchView.this.searchButtonClicked(); }});
		this.fileContentTextField.setToolTipText(I18N.translate("ToolTipSearchInWorkspaceFileContentTextField"));
		this.fileContentTextField.setEnabled(false);

		this.searchAction = new AbstractAction(I18N.translate("ActionSearchInWorkspace"), new ImageIcon("./resources/images/Search.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SearchView.this.searchButtonClicked(); }};
		this.searchAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchInWorkspace"));

		this.clearAction = new AbstractAction(I18N.translate("ActionSearchInWorkspaceClearTextField"), new ImageIcon("./resources/images/Clear.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SearchView.this.clearButtonClicked(); }};
		this.clearAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchInWorkspaceClearTextField"));

		this.pickAction = new AbstractAction(I18N.translate("ActionSearchInWorkspacePick"), new ImageIcon("./resources/images/Open.png"))
		{	@Override public void actionPerformed(ActionEvent e){ SearchView.this.pickButtonClicked(); }};
		this.pickAction.putValue(Action.SHORT_DESCRIPTION, I18N.translate("ToolTipSearchInWorkspacePick"));


		JButton searchButton = new JButton(this.searchAction);
		searchButton.setHideActionText(true);

		JButton clearButton = new JButton(this.clearAction);
		clearButton.setHideActionText(true);

		JButton pickButton = new JButton(this.pickAction);
		pickButton.setHideActionText(true);

		JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		buttonPanel.add(searchButton);
		buttonPanel.add(clearButton);

		GridBagPanel mainPanel = new GridBagPanel(new Insets(0, 5, 0, 5));
		mainPanel.add(new JLabel(I18N.translate("LabelSearchSIPName")),	3,    1,	GridBagConstraints.EAST);
		mainPanel.add(this.fileNameTextField,							3,    2,	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL, 1, 0);
		mainPanel.add(new JLabel(I18N.translate("LabelSearchMetadata")),3,    3,	GridBagConstraints.EAST);
		mainPanel.add(this.metadataTextField,							3,    4,	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL, 1, 0);
//		mainPanel.add(new JLabel(I18N.translate("LabelSearchContent")),	3,    5,	GridBagConstraints.EAST);
//		mainPanel.add(this.fileContentTextField,						3,    6,	GridBagConstraints.WEST,		GridBagConstraints.HORIZONTAL, 1, 0);

		mainPanel.add(buttonPanel,										7, 7, 1, 4,	GridBagConstraints.WEST);
		mainPanel.add(new JScrollPane(this.searchResultTable),			8, 8, 1, 4,	GridBagConstraints.CENTER,		GridBagConstraints.BOTH, 1, 1);
		mainPanel.add(this.rowCountLabel,								9,    1,	GridBagConstraints.NORTHWEST);
		mainPanel.add(pickButton,										9,    4,	GridBagConstraints.EAST);

		this.add(mainPanel, BorderLayout.CENTER);

		this.setPreferredSize(StartScreenSize);
		this.pack();
		this.setLocation(launcherView.getLocation().x + 50, launcherView.getLocation().y + 50);

		this.enableOrDisableActions();
	}

	//	========	Static Public			=======================================================

	static public void open(LauncherView launcherView)
	{
		if (SearchView == null)		SearchView = new SearchView(launcherView);

		SearchView.metadataTextField.requestFocusInWindow();
		SearchView.setVisible(true);
	}


	static public void closeAndDispose()
	{
		if (SearchView != null)
		{
			SearchView.setVisible(false);
			SearchView.dispose();
		}
	}

	//	========	Static Private			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Public			=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------

	private void tableSelectionChanged()
	{
		this.enableOrDisableActions();
	}


	private void searchButtonClicked()
	{
		new SwingWorker<Integer, Object>()
		{
			@Override public Integer doInBackground()
			{
				SearchView.this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				SmallPeskyMessageWindow waitWindow = SmallPeskyMessageWindow.openBlocking(SearchView.this, "...");

				try
				{
					String fileNamePattern = SearchView.this.fileNameTextField.getText();
					String searchString = SearchView.this.metadataTextField.getText();

					if (searchString.isEmpty())			return 0;
					if (fileNamePattern.isEmpty())		fileNamePattern = "*";

					final Pattern filePattern = new Pattern(fileNamePattern);
					String[] matchingSIPNames =
						new File(LauncherView.SIPDirectory).list(new FilenameFilter()
							{ @Override public boolean accept(File workspaceFolder, String fileName)
								{ return
										!fileName.startsWith(".")
									&&	!fileName.endsWith(Document.LockFileSuffix)
									&&	filePattern.match(fileName); }});

					Document doc = null;
					List<SearchResultElement> searchResult = new ArrayList<SearchResultElement>(matchingSIPNames.length);
					for (String matchingSIPName: matchingSIPNames)
					{
						Logger.debug("Searching in packet: " + matchingSIPName);
						waitWindow.setText(matchingSIPName);

						//	Skip folder that doesn't contain a mets.xml file:
						String sipName = LauncherView.SIPDirectory + File.separator + matchingSIPName;
						if (	 new File(sipName).isDirectory()
							&&	!new File(sipName + File.separator + Document.DefaultMetsFileName).exists())		continue;

						try
						{
							doc = Document.openReadOnly(sipName, LauncherView.Operator);

							List<NodeAbstract> nodes = doc.searchFor(searchString);
							if (nodes.isEmpty())		continue;

							for (NodeAbstract node: nodes)		searchResult.add(new SearchResultElement(matchingSIPName, node));
						}
						catch (Exception ex)
						{
							ex.printStackTrace();
							continue;
						}
						finally
						{
							if (doc != null)
								try { doc.cleanupWorkingCopy(); }
								catch (Exception ex) {}
						}
					}

					((SearchTableModel)SearchView.this.searchResultTable.getModel()).setList(searchResult);
				}
				finally
				{
					waitWindow.close();
					SearchView.this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

					//	Don't know why but under Windows the SearchView loses the focus after the search:
					SearchView.this.toFront();
				}

				SearchView.this.rowCountLabel.setText("" + ((SearchTableModel)SearchView.this.searchResultTable.getModel()).getRowCount());
				SearchView.this.enableOrDisableActions();

				return 0;
			}
		}.execute();
	}


	private void clearButtonClicked()
	{
		this.fileNameTextField.setText("");
		this.metadataTextField.setText("");
		this.fileContentTextField.setText("");

		((SearchTableModel)this.searchResultTable.getModel()).clearList();
		this.rowCountLabel.setText("");

		this.enableOrDisableActions();
	}


	private void pickButtonClicked()
	{
		int selectionIndex = this.searchResultTable.getSelectedRow();
		if (selectionIndex == -1)		return;

		int translatedIndexAfterSorting = this.searchResultTable.convertRowIndexToModel(selectionIndex);
		SearchResultElement pickedElement = ((SearchTableModel)this.searchResultTable.getModel()).get(translatedIndexAfterSorting);

		this.launcherView.openSIPInWorkspace(pickedElement.sipName, false, pickedElement.node.getAdmId());
	}

	//	--------		Actions				-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------

	private void enableOrDisableActions()
	{
		this.pickAction.setEnabled(this.searchResultTable.getSelectedRow() != -1);
	}

	//	---------		Misc				-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	public class SearchTableModel extends AbstractTableModel
	{
		List<SearchResultElement>		searchResult = new ArrayList<SearchResultElement>();


		public void setList(List<SearchResultElement> searchResult)
		{
			this.searchResult = searchResult;
			this.fireTableDataChanged();
		}


		public void clearList()
		{
			this.searchResult.clear();
			this.fireTableDataChanged();
		}


		public SearchResultElement get(int rowIndex)
		{
			return this.searchResult.get(rowIndex);
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		@Override
		public int getRowCount()
		{
			return this.searchResult.size();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		@Override
		public int getColumnCount()
		{
			return 2;
		}


		/**
		 * {@inheritDoc}
		 */
		@Override
		public String getColumnName(int columnIndex)
		{
			switch (columnIndex)
			{
				case 0:		return I18N.translate("HeaderSearchInWorkspaceResultSIP");
				case 1:		return I18N.translate("HeaderSearchInWorkspaceResultFile");
			}

			return null;
		}


		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		@Override
		public Object getValueAt(int rowIndex, int columnIndex)
		{
			SearchResultElement resultElement = this.searchResult.get(rowIndex);

			switch (columnIndex)
			{
				case 0:		return resultElement.sipName;
				case 1:		return ((NodeAbstract)resultElement.node.getTreePath().getLastPathComponent()).getPathString();
			}

			return null;
		}
	}



	private class SearchResultElement
	{
		private String			sipName;
		private NodeAbstract	node;
//		private TreePath		nodeTreePath;


		public SearchResultElement(String sipName, NodeAbstract node)
		{
			super();
			this.sipName = sipName;
			this.node = node;
//			this.nodeTreePath = node.getTreePath();
		}
	}

}
