/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.mets.metshdr;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import ch.docuteam.docudarc.common.NodeAbstract;
import ch.docuteam.docudarc.mets.Document;
import ch.docuteam.docutools.string.DateFormatter;




/**
 * This class, used by the class <a href="../Document.html">Document</a>, represents the METS header.
 * It contains the creation date, the last modification date, a status, and a collection of <a href="./Agent.html">Agent</a>s.
 *
 * @author denis
 *
 */
public class Header extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected String				createdDate;
	protected String				recordStatus;
	protected String				lastModificationDate;
	protected List<Agent>			agents = new ArrayList<Agent>();

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	protected Header() {}

	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read.
	 */
	@SuppressWarnings("unchecked")
	private Header(Document document)
	{
		this.document = document;
		this.element = (Element)this.document.selectSingleNode("./METS:mets/METS:metsHdr");

		//	Check if a header exists at all - if not, create one and place it at the right position (as the 1st element):
		if (this.element == null)
		{
			this.element = DocumentHelper.createElement("METS:metsHdr");
			this.element.addAttribute("CREATEDATE", DateFormatter.getCurrentDateTimeString());
			this.element.addAttribute("LASTMODDATE", DateFormatter.getCurrentDateTimeString());
			this.element.addAttribute("RECORDSTATUS", "New");
			this.document.getRootElement().content().add(0, this.element);
		}

		this.createdDate = this.element.attributeValue("CREATEDATE");
		this.lastModificationDate = this.element.attributeValue("LASTMODDATE");
		this.recordStatus = this.element.attributeValue("RECORDSTATUS");

		this.agents.addAll(Agent.parse(this));
	}

	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read.
	 */
	static public Header parse(Document document)
	{
		return new Header(document);
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getCreatedDate()
	{
		return this.createdDate;
	}

	public String getRecordStatus()
	{
		return this.recordStatus;
	}

	public void setRecordStatus(String string)
	{
		this.recordStatus = string;
		this.element.addAttribute("RECORDSTATUS", string);
		this.document.setIsModified();
	}

	public String getLastModificationDate()
	{
		return this.lastModificationDate;
	}

	public void setLastModificationDateToNow()
	{
		this.setLastModificationDate(DateFormatter.getCurrentDateTimeString());
	}

	public void setLastModificationDate(String currentDateTimeString)
	{
		this.lastModificationDate = currentDateTimeString;
		this.element.addAttribute("LASTMODDATE", currentDateTimeString);
		this.document.setIsModified();
	}

	public List<Agent> getAgents()
	{
		return this.agents;
	}


	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder("\n[Header:")
			.append(this.createdDate)
			.append("/")
			.append(this.lastModificationDate)
			.append("/")
			.append(this.recordStatus)
			.append("(agents:")
			.append( this.agents.size())
			.append(")");
		for (Agent agent: this.agents)		buf.append("\n\t" + agent.toString());
		buf.append("\n]");

		return buf.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

}
