/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.premis;

import java.util.ArrayList;
import java.util.List;

import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Namespace;
import org.dom4j.Node;
import org.dom4j.QName;

import uk.gov.nationalarchives.droid.core.interfaces.IdentificationResult;
import ch.docuteam.docudarc.common.NodeAbstract;
import ch.docuteam.docudarc.mets.amdsec.DigiprovWithPremis;
import ch.docuteam.docutools.file.FileChecksumCalculator;
import ch.docuteam.docutools.file.MetadataProviderDROID;
import ch.docuteam.docutools.id.UniqueID;

/**
 * This class represents a METS digiprovMD PREMIS Object.
 *
 * @author denis
 *
 */
public class Object extends NodeAbstract
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	private static final String			CompositionLevel = "0";
	private static final String			MessageDigestAlgorithm = "MD5";
	private static final String			FormatRegistryName = "PRONOM";
	private static final String			IDType = "Docuteam";
	private static final String			RelationshipType = "derivation";
	private static final String			RelationshipSubType = "derived from";
	private static final String			Xmlns_xsi = "http://www.w3.org/2001/XMLSchema-instance";
	private static final String			Xmlns_xlink = "http://www.w3.org/1999/xlink";
	private static final String			Xlink_type = "simple";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected String		type = "";
	protected String		idType = IDType;

	protected String		compositionLevel = CompositionLevel;
	protected String		messageDigestAlgorithm = MessageDigestAlgorithm;
	protected String		messageDigest = "";
	protected String		size = "";
	protected String		formatName = "";
	protected String		formatRegistryName = FormatRegistryName;
	protected String		formatRegistryKey = "";
	protected String		originalName = "";

	protected String		relationshipType = RelationshipType;
	protected String		relationshipSubType = RelationshipSubType;
	protected String		relatedObjectIdType = IDType;
	protected String		relatedObjectIdValue = "";
	protected String		relatedEventIdType = IDType;
	protected String		relatedEventIdValue = "";

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Constructors Public		=======================================================

	/**
	 * 	This constructor is used only when a new digiprov object is created programmatically
	 */
	@SuppressWarnings("unchecked")
	public Object(DigiprovWithPremis parent, Element parentElement, ch.docuteam.docudarc.mets.structmap.NodeAbstract node)
	{
		this.parent = parent;
		this.document = parent.getDocument();

		this.id = UniqueID.getXML();
		this.originalName = node.getLabel();

		this.type = (node.isFile()? "PREMIS:file": "PREMIS:representation");

		//	Insert the newly created Object *BEFORE* the first PREMISEvent, that is behind the *LAST* existing PREMISObject:
		this.element = DocumentHelper.createElement("PREMIS:object").addAttribute(new QName("type", new Namespace("xsi", Xmlns_xsi)), this.type);
		parentElement.content().add(this.indexOfLastObject(parentElement) + 1, this.element);
		//	Note: this "+ 1" works even when no objects are found: in this case, -1 is returned. When I add 1 to it, I get 0.
		Element objectIdentifier = this.element.addElement("PREMIS:objectIdentifier");
		objectIdentifier.addElement("PREMIS:objectIdentifierType").addText(this.idType);
		objectIdentifier.addElement("PREMIS:objectIdentifierValue").addText(this.id);

		if (node.isFile())
		{
			this.size = node.getSize().toString();
			try
			{
				this.messageDigest = FileChecksumCalculator.getMD5(node.getAbsolutePathString());
			}
			catch (java.lang.Exception e)
			{
				ch.docuteam.docutools.exception.Exception.remember(e);
			}

			IdentificationResult identificationResult = null;
			try
			{
				identificationResult = MetadataProviderDROID.getIdentificationResult(node.getAbsolutePathString());
			}
			catch (Exception ex){}		//	Ignore errors

			//	It can actually happen that fileFormatHit is null:
			if (identificationResult != null)
			{
				this.formatName = identificationResult.getName();
				this.formatRegistryKey = identificationResult.getPuid();
			}

			if (this.formatName == null)		this.formatName = "";
			if (this.formatRegistryKey == null)	this.formatRegistryKey = "";

			Element objectCharacteristics = this.element.addElement("PREMIS:objectCharacteristics");
			objectCharacteristics.addElement("PREMIS:compositionLevel").addText(this.compositionLevel);
			Element fixity = objectCharacteristics.addElement("PREMIS:fixity");
			fixity.addElement("PREMIS:messageDigestAlgorithm").addText(this.messageDigestAlgorithm);
			fixity.addElement("PREMIS:messageDigest").addText(this.messageDigest);
			objectCharacteristics.addElement("PREMIS:size").addText(this.size);
			Element format = objectCharacteristics.addElement("PREMIS:format");
			Element formatDesignation = format.addElement("PREMIS:formatDesignation");
			formatDesignation.addElement("PREMIS:formatName").addText(this.formatName);
			Element formatRegistry = format.addElement("PREMIS:formatRegistry");
			formatRegistry.addElement("PREMIS:formatRegistryName").addText(this.formatRegistryName);
			formatRegistry.addElement("PREMIS:formatRegistryKey").addText(this.formatRegistryKey);
		}

		this.element.addElement("PREMIS:originalName")
			.addAttribute(new QName("type", new Namespace("xlink", Xmlns_xlink)), Xlink_type)
			.addText(this.originalName);
	}

	//	========	Constructors Private	=======================================================

	/**
	 * 	This constructor is used only when a METS-File is being read
	 */
	private Object(DigiprovWithPremis parent, Element objectElement)
	{
		this.document = parent.getDocument();
		this.parent = parent;
		this.element = objectElement;

		this.type = objectElement.attributeValue("type");

		this.idType = this.getElementText("./PREMIS:objectIdentifier/PREMIS:objectIdentifierType", true);
		this.id = this.getElementText("./PREMIS:objectIdentifier/PREMIS:objectIdentifierValue", true);

		//	The following values are optional:
		this.compositionLevel = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:compositionLevel", false);
		this.messageDigestAlgorithm = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:fixity/PREMIS:messageDigestAlgorithm", false);
		this.messageDigest = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:fixity/PREMIS:messageDigest", false);
		this.size = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:size", false);
		this.formatName = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:format/PREMIS:formatDesignation/PREMIS:formatName", false);
		this.formatRegistryName = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:format/PREMIS:formatRegistry/PREMIS:formatRegistryName", false);
		this.formatRegistryKey = this.getElementText("./PREMIS:objectCharacteristics/PREMIS:format/PREMIS:formatRegistry/PREMIS:formatRegistryKey", false);
		this.originalName = this.getElementText("./PREMIS:originalName", false);

		//	The relationship is optional, but if it is present, the enclosed properties are mandatory:
		Node relationshipNode = this.element.selectSingleNode("./PREMIS:relationship");
		if (relationshipNode != null)
		{
			this.relationshipType = this.getElementText("./PREMIS:relationship/PREMIS:relationshipType", true);
			this.relationshipSubType = this.getElementText("./PREMIS:relationship/PREMIS:relationshipSubType", true);
			this.relatedObjectIdType = this.getElementText("./PREMIS:relationship/PREMIS:relatedObjectIdentification/PREMIS:relatedObjectIdentifierType", true);
			this.relatedObjectIdValue = this.getElementText("./PREMIS:relationship/PREMIS:relatedObjectIdentification/PREMIS:relatedObjectIdentifierValue", true);
			this.relatedEventIdType = this.getElementText("./PREMIS:relationship/PREMIS:relatedEventIdentification/PREMIS:relatedEventIdentifierType", true);
			this.relatedEventIdValue = this.getElementText("./PREMIS:relationship/PREMIS:relatedEventIdentification/PREMIS:relatedEventIdentifierValue", true);
		}
	}


	//	========	Static Public			=======================================================

	/**
	 * This method is used only when a METS-File is being read. Create the list of Digiprov Objects out of the given parent AMDSectionDigiprov
	 */
	static public List<Object> parse(DigiprovWithPremis parent, Element premisElement)
	{
		List<Object> fileList = new ArrayList<Object>();

		for (java.lang.Object o: premisElement.selectNodes("./PREMIS:object"))		fileList.add(new Object(parent, (Element)o));

		return fileList;
	}

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public String getIdType()
	{
		return this.idType;
	}

	public String getType()
	{
		return this.type;
	}

	public String getCompositionLevel()
	{
		return this.compositionLevel;
	}

	public String getMessageDigestAlgorithm()
	{
		return this.messageDigestAlgorithm;
	}

	public String getMessageDigest()
	{
		return this.messageDigest;
	}

	public String getSize()
	{
		return this.size;
	}

	public String getFormatName()
	{
		return this.formatName;
	}

	public String getFormatRegistryName()
	{
		return this.formatRegistryName;
	}

	public String getFormatRegistryKey()
	{
		return this.formatRegistryKey;
	}

	public String getOriginalName()
	{
		return this.originalName;
	}

	public void setOriginalName(String newOriginalName)
	{
		this.originalName = newOriginalName;
		this.element.selectSingleNode("PREMIS:originalName").setText(this.originalName);
	}

	public String getRelationshipType()
	{
		return this.relationshipType;
	}

	public String getRelationshipSubType()
	{
		return this.relationshipSubType;
	}

	public String getRelatedObjectIdType()
	{
		return this.relatedObjectIdType;
	}

	public String getRelatedObjectIdValue()
	{
		return this.relatedObjectIdValue;
	}

	public String getRelatedEventIdType()
	{
		return this.relatedEventIdType;
	}

	public String getRelatedEventIdValue()
	{
		return this.relatedEventIdValue;
	}

	public void addRelationships(Event event)
	{
		this.relatedEventIdValue = event.getId();
		this.relatedObjectIdValue = event.getLinkingObjectIdValue();

		Element relationship = this.element.addElement("PREMIS:relationship");
		relationship.addElement("PREMIS:relationshipType").addText(this.relationshipType);
		relationship.addElement("PREMIS:relationshipSubType").addText(this.relationshipSubType);
		Element relatedObjId = relationship.addElement("PREMIS:relatedObjectIdentification");
		relatedObjId.addElement("PREMIS:relatedObjectIdentifierType").addText(this.relatedObjectIdType);
		relatedObjId.addElement("PREMIS:relatedObjectIdentifierValue").addText(this.relatedObjectIdValue);
		Element relatedEventId = relationship.addElement("PREMIS:relatedEventIdentification");
		relatedEventId.addElement("PREMIS:relatedEventIdentifierType").addText(this.relatedEventIdType);
		relatedEventId.addElement("PREMIS:relatedEventIdentifierValue").addText(this.relatedEventIdValue);
	}


	public void setRelatedObject(Object object)
	{
		this.setRelatedObject(object, null, null);
	}

	public void setRelatedObject(Object object, String relationshipType, String relationshipSubType)
	{
		this.relatedObjectIdValue = object.getId();
		this.element.selectSingleNode("PREMIS:relationship/PREMIS:relatedObjectIdentification/PREMIS:relatedObjectIdentifierValue").setText(this.relatedObjectIdValue);

		if (relationshipType != null)
		{
			this.relationshipType = relationshipType;
			this.element.selectSingleNode("PREMIS:relationship/PREMIS:relationshipType").setText(this.relationshipType);
		}

		if (relationshipSubType != null)
		{
			this.relationshipSubType = relationshipSubType;
			this.element.selectSingleNode("PREMIS:relationship/PREMIS:relationshipSubType").setText(this.relationshipSubType);
		}
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------

	private int indexOfLastObject(Element parentElement)
	{
		for (int i = parentElement.content().size() - 1; i >= 0; i--)
		{
			if ("object".equals(((Node)parentElement.content().get(i)).getName()))		return i;
		}

		return -1;
	}

	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		StringBuilder stringBuilder =
		new StringBuilder("[PREMISObject:id=")
			.append(this.id)
			.append("(")
			.append(this.type)
			.append("/")
			.append(this.formatName)
			.append("/")
			.append(this.formatRegistryName)
			.append("/")
			.append(this.formatRegistryKey)
			.append("/")
			.append(this.originalName)
			.append(")");
		if ((this.relatedObjectIdValue.length() != 0) || (this.relatedEventIdValue.length() != 0))
			stringBuilder
				.append("Object->")
				.append(this.relatedObjectIdValue)
				.append("/Event->")
				.append(this.relatedEventIdValue);
		stringBuilder.append("]");
		return stringBuilder.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
