/**
 *	Copyright (C) 2011-2013 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docudarc.util;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.net.ConnectException;
import java.util.*;

import javax.swing.SwingWorker;

import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import ch.docuteam.docudarc.exceptions.*;
import ch.docuteam.docutools.file.*;
import ch.docuteam.docutools.file.exception.*;
import ch.docuteam.docutools.os.OperatingSystem;
import ch.docuteam.docutools.os.SystemProcess;
import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.string.DateFormatter;

import com.artofsolving.jodconverter.DocumentConverter;
import com.artofsolving.jodconverter.XmlDocumentFormatRegistry;
import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.connection.SocketOpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.converter.OpenOfficeDocumentConverter;


/**
 * @author denis
 *
 */
public abstract class FileConverter
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static private final String				DefaultMigrationConfigFile = "config/migration-config.xml";
	static private final String				DefaultMigrationHomeFolder = ".";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private String					MigrationConfigFile = DefaultMigrationConfigFile;
	static private String					MigrationHomeFolder = DefaultMigrationHomeFolder;

	static private org.dom4j.Document		MigrationConfigFileDocument;

	static private String					RecentlyUsedFileConverterName = "";

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	--------		Converting			-------------------------------------------------------

	static public File convertFile(File file) throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException, IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException, FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException, OOConverterShallNotBeUsedException
	{
		return convertFile(file.getPath());
	}

	static public File convertFile(File sourceFile, File destinationFolder) throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException, IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException, FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException, OOConverterShallNotBeUsedException
	{
		return convertFile(sourceFile.getPath(), destinationFolder.getPath());
	}

	static public File convertFile(String fileName) throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException, IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException, FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException, OOConverterShallNotBeUsedException
	{
		return convertFile(fileName, null);
	}

	static public File convertFile(String fileName, String destinationFolderName) throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException, IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException, FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException, OOConverterShallNotBeUsedException
	{
		//	Whatever comes in, make the fileNames absolute:
		fileName = new File(fileName).getAbsolutePath();
		if (destinationFolderName != null)		destinationFolderName = new File(destinationFolderName).getAbsolutePath();

		RecentlyUsedFileConverterName = "";

		FileWithMetadata file = new FileWithMetadata(fileName);

		if (!file.exists())
		{
			Logger.error("File not found: '" + file.getPath() + "'");
			throw new FileNotFoundException(file.getPath());
		}

		initializeIfNecessary();

		Logger.info("Converting file: " + file.getPath());

		List<?> migrationInstructions = getMigrationInstructions(file);
		if (migrationInstructions.isEmpty())
		{
			Logger.info("... no conversion instruction found.");
			return null;
		}
		if (migrationInstructions.size() >= 2)
		{
			Logger.error("Can't handle multiple conversion instructions for file '" + file.getName() + "'");
			throw new IndexOutOfBoundsException("Can't handle multiple conversion instructions for file '" + file.getName() + "'");
		}

		Element migrationInstruction = (Element)migrationInstructions.get(0);

		//	If the destination's pronomID equals the source's pronomID, don't convert:
		//	Note: file.getFormatPronomID() NEVER returns null, migrationInstruction.attributeValue("targetPronom") MIGHT return null:
		if (file.getFormatPronomID().equalsIgnoreCase(migrationInstruction.attributeValue("targetPronom")))
		{
			Logger.info("... no conversion required.");
			return null;
		}

		if (destinationFolderName != null)		FileUtil.createFolderMerging(destinationFolderName);

		Logger.info("Treating file as " + migrationInstruction.attributeValue("name") + "...");

		try
		{
			List<?> migrationSteps = migrationInstruction.selectNodes("step");
			return
				(migrationSteps.isEmpty())
				?	convertFile(file, destinationFolderName, migrationInstruction)
				:	convertFile(file, destinationFolderName, migrationSteps);
		}
		catch (InvocationTargetException ex)
		{
			//	If the cause is an OOConverterShallNotBeUsedException, throw it:
			if (ex.getCause().getClass() == OOConverterShallNotBeUsedException.class)		throw (OOConverterShallNotBeUsedException)ex.getCause();

			//	Otherwise rethrow the original exception:
			throw ex;
		}
	}

	//	--------		Accessing			-------------------------------------------------------

	public static String getRecentlyUsedFileConverterName()
	{
		return RecentlyUsedFileConverterName;
	}

	public static void setMigrationConfigFile(String newMigrationConfigFile) throws DocumentException
	{
		MigrationConfigFile = newMigrationConfigFile;
		initialize();
	}

	public static void setMigrationHomeFolder(String newMigrationHomeFolder)
	{
		MigrationHomeFolder = newMigrationHomeFolder;
	}

	//	========	Static Private			=======================================================

	static private File convertFile(FileWithMetadata file, String destinationFolderName, Element migrationInstruction) throws IOException, IllegalArgumentException, SecurityException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException, FileIsNotADirectoryException, BadPronomIdException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException, OOConverterShallNotBeUsedException
	{
		String destinationPath = calculateDestinationFilePath(file, destinationFolderName, migrationInstruction.attributeValue("targetExtension"));

		Logger.info("Destination file: " + destinationPath);

		String[] commandLine = constructCommandLine(migrationInstruction, new String[]{ file.getAbsolutePath(), destinationPath });
		executeCommandLine(commandLine);

		String currentPronomId = MetadataProviderDROID.getFileFormatPUID(destinationPath);
		String expectedPronomId = migrationInstruction.attributeValue("targetPronom");
		if ((expectedPronomId != null) && !expectedPronomId.isEmpty()
		&& !currentPronomId.equalsIgnoreCase(expectedPronomId))				throw new BadPronomIdException(expectedPronomId, currentPronomId);

		return new File(destinationPath);
	}


	static private File convertFile(FileWithMetadata file, String destinationFolderName, List<?> migrationInstructions) throws IOException, IllegalArgumentException, SecurityException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException, FileIsNotADirectoryException, BadPronomIdException, FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException, OOConverterShallNotBeUsedException
	{
		List<String> tempFilePathsToDelete = new ArrayList<String>();
		String sourcePathString = file.getAbsolutePath();
		String destinationPathString = "";
		int stepNumber = migrationInstructions.size();

		try
		{
			// Loop thru the steps:
			int stepCounter = 1;
			for (Object i: migrationInstructions)
			{
				Element migrationInstruction = (Element)i;

				//	If the destination's pronomID equals the source's pronomID, skip this step:
				//	Note: file.getFormatPronomID() NEVER returns null, migrationInstruction.attributeValue("targetPronom") MIGHT return null:
				if (file.getFormatPronomID().equalsIgnoreCase(migrationInstruction.attributeValue("targetPronom")))
				{
					Logger.info("Step " + stepCounter + "/" + stepNumber + ": No conversion required.");
					++stepCounter;
					continue;
				}

				//	Calculate destination file path for this step:
				destinationPathString = calculateDestinationFilePath(file, destinationFolderName, migrationInstruction.attributeValue("targetExtension"));

				Logger.info("Step " + stepCounter + "/" + stepNumber + ": Destination file: " + destinationPathString);

				//	Remember this intermediary file for later deletion (excluding the last step):
				if (stepCounter != stepNumber)		tempFilePathsToDelete.add(destinationPathString);

				//	Perform the conversion:
				String[] commandLine = constructCommandLine(migrationInstruction, new String[] { sourcePathString, destinationPathString });
				executeCommandLine(commandLine);

				String currentPronomId = MetadataProviderDROID.getFileFormatPUID(destinationPathString);
				String expectedPronomId = migrationInstruction.attributeValue("targetPronom");
				if ((expectedPronomId != null) && !expectedPronomId.isEmpty()
				&& !currentPronomId.equalsIgnoreCase(expectedPronomId))				throw new BadPronomIdException(expectedPronomId, currentPronomId);

				//	Set the input of the next step to be the output of this step:
				sourcePathString = destinationPathString;
				++stepCounter;
			}
		}
		finally
		{
			//	Delete all intermediary files:
			for (String path: tempFilePathsToDelete)		FileUtil.delete(path);
		}

		return new File(destinationPathString);
	}

	//	--------		Initialization		-------------------------------------------------------

	static private void initializeIfNecessary() throws DocumentException
	{
		if (MigrationConfigFileDocument == null)		initialize();
	}


	static private void initialize() throws DocumentException
	{
		Logger.info("Initializing File Migration, from " + MigrationConfigFile + "...");

		// Parse MigrationConfig-File:
		MigrationConfigFileDocument = new SAXReader().read(new File(MigrationConfigFile));
	}

	//	--------		Tools				-------------------------------------------------------

	static private String calculateDestinationFilePath(File file, String destinationFolderName, String newExtension)
	{
		String oldFileName = file.getName();
		String newFileName = FileUtil.asFileNameWithoutExtension(oldFileName);
		String oldExtension = FileUtil.asFileNameExtension(oldFileName);
		//	If new and old extension are the same, make it like 'file_pdf.pdf':
		if (oldExtension.equalsIgnoreCase(newExtension))					newFileName += "_" + oldExtension;

		//	Check if the destination file already exists. If yes, append the current timestamp to the filename:
		String destinationFilePath = (destinationFolderName == null? file.getParentFile().getAbsolutePath(): destinationFolderName) + "/" + newFileName;
		if (new File(destinationFilePath + "." + newExtension).exists())	destinationFilePath += "_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);
		destinationFilePath += "." + newExtension;

		return destinationFilePath;
	}


	static private List<?> getMigrationInstructions(FileWithMetadata file)
	{
		Logger.info("Looking for PUID " + file.getFormatPronomID() + "...");
		List<?> instructions = MigrationConfigFileDocument.selectNodes("//puid[@name='" + file.getFormatPronomID() + "']");
		if (!instructions.isEmpty())		return instructions;

		Logger.info("Looking for mimetype " + file.getMimeType() + "...");
		instructions = MigrationConfigFileDocument.selectNodes("//mimeType[@name='" + file.getMimeType() + "']");
		if (!instructions.isEmpty())		return instructions;

		String fileNameExtension = file.getName().substring(file.getName().lastIndexOf(".") + 1);
		Logger.info("Looking for extension " + fileNameExtension + "...");
		return MigrationConfigFileDocument.selectNodes("//extension[@name='" + fileNameExtension + "']");
	}


	static private String[] constructCommandLine(Element migrationInstruction, String[] args)
	{
		final String Separator = "#";

		Element applicationElement = (Element) MigrationConfigFileDocument.selectSingleNode("//application[@id='" + migrationInstruction.attributeValue("applicationID") + "']");
		String commandLine = applicationElement.attributeValue("executable");
		RecentlyUsedFileConverterName = applicationElement.attributeValue("name");

		String applicationParameter = applicationElement.attributeValue("parameter");
		String instructionParameter = migrationInstruction.attributeValue("parameter");

		//	instructionParameter take precedence over applicationParameter:
		if (instructionParameter != null && !instructionParameter.trim().isEmpty())
			commandLine += Separator + instructionParameter;
		else if (applicationParameter != null && !applicationParameter.trim().isEmpty())
			commandLine += Separator + applicationParameter;

		int argIndex = 0;
		while (true)
		{
			String argPlaceholderString = "{[arg" + (argIndex + 1) + "]}";
			if (commandLine.indexOf(argPlaceholderString) == -1)		break;

			commandLine = commandLine.replace(argPlaceholderString, args[argIndex]);
			argIndex = argIndex + 1;
		}

		//	Any possible additional arguments are appended to the commandLine:
		for (int i = argIndex; i < args.length; i++)
		{
			if (!args[i].isEmpty())		commandLine = commandLine + Separator + args[i];
		}

		return commandLine.trim().split(Separator);
	}


	/**
	 * @throws OOConverterShallNotBeUsedException The call to the converter goes via reflection, that's why this Exception is not visible. It can be thrown by the OOConverter.
	 */
	static private void executeCommandLine(String[] commandLine) throws IllegalArgumentException, SecurityException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, ClassNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException, FileConversionException, OOConverterShallNotBeUsedException
	{
		Logger.info("Executing command line: " + Arrays.toString(commandLine));

		if (commandLine[0].startsWith("Class:"))
		{
			//	Application is a Class:
			//	Cut away leading "Class:" from class name:
			String className = commandLine[0].substring(6);

			//	Cut away 1st element of commandLine (= class name):
			commandLine = Arrays.copyOfRange(commandLine, 1, commandLine.length);

			Logger.debug("Conversion-process starting");
			Object result = Class.forName(className).getMethod("main", commandLine.getClass()).invoke(null, new Object[] { commandLine });
			Logger.info("Conversion-process finished" + ((result != null)? ": " + result: ""));
		}
		else
		{
			int errorCode = SystemProcess.execute(MigrationHomeFolder, commandLine);
			Logger.info("Conversion-process finished: " + errorCode);

			if (errorCode != 0)		throw new FileConversionException(errorCode);
		}
	}

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	//	To start the OpenOffice service headless:
	//	Windows:
	//		soffice.exe -headless -nofirststartwizard -accept="socket,host=localhost,port=8100;urp;StarOffice.Service"
	//	OS X:
	//		/Applications/OpenOffice.org.app/Contents/MacOS/soffice.bin -headless -nofirststartwizard -accept="socket,host=localhost,port=8100;urp;StarOffice.Service" &
	/**
	 * This class is a wrapper around the JODConverter class, which in turn is a wrapper around the OpenOffice Document Converter class.
	 * This class is used by the FileConverter reflectively, hence it seems to the compiler as if it is unused.
	 * The only method used from the outside is static public void main().
	 */
	static public abstract class OOConverter
	{
		//	Several different default values:
		static private final String				DefaultPathBase = "apps";
		static private final String				DefaultPathMac = DefaultPathBase + "/OpenOffice.org.app";
		static private final String				DefaultPathWindows = DefaultPathBase + "/OpenOffice.org 3";
		static private final String				PathRelativeMac = "/Contents/MacOS/soffice.bin";
		static private final String				PathRelativeWindows = "/program/soffice.exe";

		static private final String				DefaultHost = "localhost";
		static private final int				DefaultPort = 8100;

		static private final String				DefaultJODDocumentFormatsFile = "config/document-formats.xml";

		static private final int				DefaultNumberOfInitializationRetries = 10;


		static private String					Path = OperatingSystem.isMacOSX()? DefaultPathMac: DefaultPathWindows;
		static private String					Host = DefaultHost;
		static private int						Port = DefaultPort;

		static private String					JODDocumentFormatsFile = DefaultJODDocumentFormatsFile;

		static private boolean					ShallNotBeUsed = false;
		static private boolean					IsInitializing = false;
		static private int						NumberOfInitializationRetries = DefaultNumberOfInitializationRetries;

		static private OpenOfficeConnection		Connection;
		static private DocumentConverter		Converter;
		static private Process					OOProcess;

		//	========	Static Initializer		=======================================================

		//	========	Main					=======================================================

		/**
		 * This main method is the gateway to run the converter. It's NOT just for tests! Don't delete it!
		 * @throws OOConverterShallNotBeUsedException
		 */
		static public void main(String... args) throws IOException, FileUtilExceptionListException, OOConverterShallNotBeUsedException
		{
			if (args.length != 2)
			{
				System.err.println("ERROR: Wrong number of arguments.");
				System.err.println("");
				System.err.println("Usage: FileConverter$OOConverter [path/to/]sourceFile [path/to/]destinationFile");
				return;
			}

			convert2PDF(args[0], args[1]);
		}

		//	========	Static Public			=======================================================

		/**
		 * This must be called BEFORE the OOConverter is initialized!
		 */
		public static void setJODDocumentsFormatFile(String newJODDocumentFormatsFile)
		{
			JODDocumentFormatsFile = newJODDocumentFormatsFile;
		}

		/**
		 * This must be called BEFORE the OOConverter is initialized!
		 */
		public static void setNumberOfInitializationRetries(Integer newNumberOfInitializationRetries)
		{
			if (newNumberOfInitializationRetries == null)		return;
			if (newNumberOfInitializationRetries <= 0)			return;

			NumberOfInitializationRetries = newNumberOfInitializationRetries;
		}

		static public Integer getNumberOfInitializationRetries()
		{
			return NumberOfInitializationRetries;
		}

		static public String getConverterPath()
		{
			return Path;
		}

		static public boolean shallNotBeUsed()
		{
			return ShallNotBeUsed;
		}


		/**
		 * Initialize in a separate thread.
		 */
		static public void initializeDontWait(String ooConverterPath, String ooConverterHost, Integer ooConverterPort)
		{
			if (!(ooConverterHost == null || ooConverterHost.isEmpty()))		Host = ooConverterHost;
			if (!(ooConverterPort == null))										Port = ooConverterPort;
			initializeDontWait(ooConverterPath);
		}


		/**
		 * Initialize in a separate thread.
		 */
		static public void initializeDontWait(String ooConverterPath)
		{
			if (!(ooConverterPath == null || ooConverterPath.isEmpty()))		Path = ooConverterPath;
			initializeDontWait();
		}


		/**
		 * Initialize in a separate thread.
		 */
		static public void initializeDontWait()
		{
			new SwingWorker<Integer, Object>()
			{
				@Override
				public Integer doInBackground()
				{
					initialize();

					return 0;
				}
			}.execute();
		}


		static public void initialize(String ooConverterPath, String ooConverterHost, Integer ooConverterPort)
		{
			if (!(ooConverterHost == null || ooConverterHost.isEmpty()))		Host = ooConverterHost;
			if (!(ooConverterPort == null))										Port = ooConverterPort;
			initialize(ooConverterPath);
		}


		static public void initialize(String ooConverterPath)
		{
			if (!(ooConverterPath == null || ooConverterPath.isEmpty()))		Path = ooConverterPath;
			initialize();
		}


		static public void initialize()
		{
			//	Maybe the Converter exists already?
			if (Converter != null)		return;

			//	Converter does not exist. But maybe the OO service is just being initialized in another thread? If yes, wait until initialization is finished:
			while(IsInitializing)
			{
				Logger.debug("Waiting for initialization to finish...");
				try { Thread.sleep(500); } catch (InterruptedException x){}
			}

			//	Maybe the Converter exists now?
			if (Converter != null)		return;

			//	Try to create the converter - if it succeeded, the OO Service was already running:
			Converter = createConverter();
			if (Converter != null)		return;

			String ooConverterPath = Path + (OperatingSystem.isMacOSX()? PathRelativeMac: PathRelativeWindows);

			Logger.debug("Starting OO Service on host:'" + Host + "' port:'" + Port + "' path:'" + ooConverterPath + "'");

			//	The OO Service is not running, so start it:
			try
			{
				//	This is when initializing in an own thread:
				IsInitializing = true;

				try
				{
					//	Is the OO application where it's supposed to be?
					if (!new File(ooConverterPath).exists())
					{
						if (Path.startsWith(DefaultPathBase))
						{
							Logger.warn("Could not find the default OO application for converting files in folder: " + Path);

							//	The OO path is the defaultPath but OO is not there - so I suppose that OO shall not be used at all:
							ShallNotBeUsed = true;
							return;
						}

						System.out.println("I was told that the OO application is right there.");
						System.out.println("Maybe this location is wrong or the OO application is not installed there?");

						throw new FileNotFoundException(ooConverterPath);
					}

					//	Start the OOProcess:
					OOProcess = Runtime.getRuntime().exec(new String[]{ooConverterPath, "-headless", "-nofirststartwizard", "-accept=socket,host=" + Host + ",port=" + Port + ";urp;StarOffice.Service"});
				}
				catch (IOException x)
				{
					x.printStackTrace();
					return;
				}

				//	Now that the OOProcess is running, create the Converter, retry several times, wait 1 sec between each try:
				for (int i = NumberOfInitializationRetries; i > 0; i--)
				{
					Logger.debug(i + ":");

					Converter = createConverter();
					if (Converter != null)
					{
						Logger.debug("OOConverter established");
						break;
					}

					//	Wait a sec and then retry:
					try { Thread.sleep(1000); } catch (InterruptedException x){}
				}
			}
			finally
			{
				IsInitializing = false;
			}
		}


		static public void disconnect()
		{
			if (Connection != null)
			{
				Logger.debug("Disconnecting OOConverter...");

				//	Sometimes, connection.disconnect() throws a NullPointerException for no obvious reason:
				try
				{
					Connection.disconnect();
				}
				catch(NullPointerException x) {}
				finally
				{
					Connection = null;
				}
			}

			if (OOProcess != null)
			{
				Logger.debug("Destroying OOProcess...");

				OOProcess.destroy();

				//	Wait a sec (otherwise it won't work):
				try { Thread.sleep(1000); } catch (InterruptedException x){}
			}

			//	Clean-up thoroughly:
			Path = OperatingSystem.isMacOSX()? DefaultPathMac: DefaultPathWindows;
			Host = DefaultHost;
			Port = DefaultPort;

			JODDocumentFormatsFile = DefaultJODDocumentFormatsFile;

			NumberOfInitializationRetries = DefaultNumberOfInitializationRetries;

			Converter = null;
			ShallNotBeUsed = false;
		}


		static public void convert2PDF(String sourceFileName, String destinationFileName) throws IOException, FileUtilExceptionListException, OOConverterShallNotBeUsedException
		{
			initializeIfNecessary();
			if (ShallNotBeUsed)		throw new OOConverterShallNotBeUsedException();

			Logger.info("Converting file: '" + sourceFileName + "' to PDF: '" + destinationFileName + "'");

			File sourceFile = new File(sourceFileName);
			File destinationFile = new File(destinationFileName);

			if (!sourceFile.exists())		throw new FileNotFoundException(sourceFileName);
			FileUtil.createFolderMerging(destinationFile.getParentFile());
			FileUtil.delete(destinationFile);

			try
			{
				Converter.convert(sourceFile, destinationFile);
			}
			catch (NullPointerException x)
			{
				Logger.error("File conversion failed - maybe the OpenOffice service is not running?");
				throw x;
			}
		}


		static public boolean isInstalledLocallyForWindows()
		{
			return new File(DefaultPathWindows).exists();
		}

		static public boolean isInstalledLocallyForOSX()
		{
			return new File(DefaultPathMac).exists();
		}

		/**
		 * Return the remote path for Windows if the current OS is Windows and the path is not the local default path.
		 * Otherwise return an empty string.
		 * @return
		 */
		static public String getRemotePathForWindows()
		{
			return OperatingSystem.isWindows() && !Path.equals(DefaultPathWindows)? Path: "";
		}

		/**
		 * Return the remote path for OSX if the current OS is OSX and the path is not the local default path.
		 * Otherwise return an empty string.
		 * @return
		 */
		static public String getRemotePathForOSX()
		{
			return OperatingSystem.isMacOSX() && !Path.equals(DefaultPathMac)? Path: "";
		}

		//	========	Static Private			=======================================================

		static private void initializeIfNecessary()
		{
			if (Converter == null)		initialize();
		}


		static private DocumentConverter createConverter()
		{
			DocumentConverter converter = null;

			try
			{
				Logger.debug("Connecting to OO Service...");

				(Connection = new SocketOpenOfficeConnection(Port)).connect();
				try
				{
					Logger.debug("Trying to initialize OO Service with initialization file '" + JODDocumentFormatsFile + "'");
					converter = new OpenOfficeDocumentConverter(Connection, new XmlDocumentFormatRegistry(new FileInputStream(JODDocumentFormatsFile)));
				}
				catch (FileNotFoundException e)
				{
					Logger.debug("Initialization file '" + JODDocumentFormatsFile + "' not found, using defaults");
					converter = new OpenOfficeDocumentConverter(Connection);
				}

				Logger.debug("... OO Service connected");
			}
			catch (ConnectException e)
			{
				Logger.debug(e.toString());
			}

			return converter;
		}
	}


	/**
	 * This class is a dummy conversion class that simply creates a copy of a file.
	 * This class is used by the FileConverter reflectively, hence it seems to the compiler as if it is unused.
	 * The only method used from the outside is static public void main().
	 */
	@SuppressWarnings("unused")
	static private class Copy
	{
		//	========	Static Public			=======================================================

		/**
		 * This main method is the gateway to run the converter. It's NOT just for tests! Don't delete it!
		 */
		static public void main(String... args) throws IOException, FileUtilExceptionListException
		{
			if (args.length != 2)
			{
				System.err.println("ERROR: Wrong number of arguments.");
				System.err.println("");
				System.err.println("Usage: FileConverter$OOConverter [path/to/]sourceFile [path/to/]destinationFile");
				return;
			}

			FileUtil.copyToOverwriting(args[0], args[1]);
		}
	}

}
